<?php

namespace App\Services;

use App\Enums\InterestPlan;
use App\Enums\StatusEnum;
use App\Enums\TransactionType;
use App\Models\InvestmentScheme;
use App\Models\PaymentMethod;
use App\Models\Referral;
use App\Models\SystemSettings;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use League\Csv\Writer;

class AdminService {

    public static function dashboard() {
        $adminDashboard = [];

        $pendingDepositsUSDSum = Transaction::where( 'status', StatusEnum::WAITING )
        ->whereIn( 'type', [ TransactionType::Deposit, TransactionType::Investment ] )
        ->sum( 'amount_usd' );

        $approvedDepositsUSDSum = Transaction::where( 'status', StatusEnum::APPROVED )
        ->whereIn( 'type', [ TransactionType::Deposit, TransactionType::Investment ] )
        ->sum( 'amount_usd' );

        $pendingWithdrawalSum = Transaction::where( 'status', StatusEnum::WAITING )
        ->where( 'type', TransactionType::Withdrawal )
        ->sum( 'amount_usd' );

        $approvedWithdrawalSum = Transaction::where( 'status', StatusEnum::APPROVED )
        ->where( 'type', TransactionType::Withdrawal )
        ->sum( 'amount_usd' );

        $adminDashboard[ 'pending_deposits_sum' ] = $pendingDepositsUSDSum;
        $adminDashboard[ 'approved_deposits_sum' ] = $approvedDepositsUSDSum;
        $adminDashboard[ 'pending_withdrawal_sum' ] = $pendingWithdrawalSum;
        $adminDashboard[ 'approved_withdrawal_sum' ] = $approvedWithdrawalSum;
        $adminDashboard[ 'total_users' ] = User::count();
        $adminDashboard[ 'total_referrals' ] = Referral::count();
        $adminDashboard[ 'total_investments_schemes' ] = InvestmentScheme::count();

        return $adminDashboard;
    }

    public static function schemes() {
        $schemes = InvestmentScheme::withCount(['investments as total_investors' => function ($query) {
            $query->whereHas('transaction', function ($transactionQuery) {
                $transactionQuery->where('status', StatusEnum::APPROVED);
            });
        }])
        ->orderBy('status', 'asc')
        ->get();

        // After retrieving the schemes, count distinct users from the investments
        foreach ($schemes as $scheme) {
            $scheme->total_investors = $scheme->investments()
                ->whereHas('transaction', function ($transactionQuery) {
                    $transactionQuery->where('status', StatusEnum::APPROVED);
                })
                ->distinct('transaction_id') // Count distinct transactions associated with users
                ->count('user_id'); // Count distinct user IDs
        }

        return $schemes;
    }

    public static function paymentMethods(){
        $payment_methods = PaymentMethod::all();
        return $payment_methods;
    }

    public static function updateSettings(array $settings){
        return SystemSettings::updateSettings($settings);
    }

    public static function transactions(array $filters = [], $perPage = 10000)  {
        $query = Transaction::select('transactions.*',
            DB::raw("CONCAT(users.first_name, ' ', users.last_name) as investor_name"))
            ->join('users', 'users.id', '=', 'transactions.user_id')
            ->where('type', '!=', TransactionType::Earnings)
            ->selectRaw("users.email as email");

        if (!empty($filters)) {
            foreach ($filters as $key => $value) {
                if (in_array($key, (new Transaction())->getFillable())) {
                    $query->where($key, $value);
                }
            }
        }

        $query->orderBy('transactions.created_at', 'desc');
        return $query->paginate($perPage);
    }


    public static function users(array $filters = [], $perPage = 1000) {
        $isLarge = $perPage > 500;

        // Only cache small result sets
        if ($isLarge) {
            return self::buildUsersQuery($filters)->cursorPaginate($perPage);
        }

        $cacheKey = 'cached_users_' . md5(json_encode($filters) . "_$perPage");

        try {
            return Cache::remember($cacheKey, now()->addMinutes(2), function () use ($filters, $perPage) {
                return self::buildUsersQuery($filters)->paginate($perPage);
            });
        } catch (\Exception $e) {
            Log::error('Failed to cache users data', [
                'error' => $e->getMessage(),
                'filters' => $filters,
                'perPage' => $perPage,
            ]);

            // Fallback to non-cached query
            return self::buildUsersQuery($filters)->paginate($perPage);
        }
    }

    private static function buildUsersQuery(array $filters) {
        $query = User::query()
            ->withTrashed()
            ->select([
                'users.*',
                DB::raw("(SELECT COALESCE(SUM(amount_usd), 0) FROM transactions
                        WHERE transactions.user_id = users.id
                        AND type = '".TransactionType::Withdrawal."'
                        AND status = '".StatusEnum::APPROVED."') as total_withdrawals_sum"),
                DB::raw("(SELECT COALESCE(SUM(amount_usd), 0) FROM transactions
                        WHERE transactions.user_id = users.id
                        AND type IN ('".TransactionType::Deposit."','".TransactionType::Investment."')
                        AND status = '".StatusEnum::APPROVED."') as total_deposits_sum"),
                DB::raw("(SELECT COALESCE(SUM(investments.total_returns), 0)
                        FROM investments
                        JOIN transactions ON transactions.id = investments.transaction_id
                        WHERE transactions.user_id = users.id
                        AND transactions.status = '".StatusEnum::APPROVED."'
                        AND investments.status = '".StatusEnum::ACTIVE."'
                        AND investments.interest_plan = '".InterestPlan::COMPOUNDING."') as compounding_sum"),
                DB::raw("(SELECT COUNT(*) FROM referrals
                        WHERE referrals.referee_id = users.id) as referral_count"),
            ]);

        foreach ($filters as $key => $value) {
            if (in_array($key, (new User())->getFillable())) {
                if ($value === null) {
                    $query->whereNull($key);
                } elseif (is_array($value)) {
                    $query->whereIn($key, $value);
                } else {
                    $query->where($key, $value);
                }
            }
        }

        return $query;
    }

    public static function getFilteredUsers($filterers, $csv = false){

        $attachments = [];
        $usersList = [];

        $baseQuery = User::select(
            'users.id',
            'users.email',
            'users.phone'
        )
        ->leftJoin( 'transactions', 'users.id', '=', 'transactions.user_id' )
        ->selectRaw( 'COALESCE(SUM(transactions.amount_usd), 0) as total_spent' )
        ->groupBy( 'users.id', 'users.email', 'users.phone' );
        // Group by all non-aggregated columns


        $filters = [
            'paying_customers' => [
                'query' => function () use ( $baseQuery ) {
                    return ( clone $baseQuery )->having( 'total_spent', '>', 0 );
                }
                ,
                'filename' => 'paying_customers_export'
            ],
            'non_paying_customers' => [
                'query' => function () use ( $baseQuery ) {
                    return ( clone $baseQuery )->having( 'total_spent', '=', 0 );
                }
                ,
                'filename' => 'non_paying_customers_export'
            ],
            'new_users' => [
                'query' => function () use ( $baseQuery ) {
                    return ( clone $baseQuery )->where( 'users.created_at', '>=', now()->subDays( 30 ) );
                }
                ,
                'filename' => 'new_users_export'
            ]
        ];

         // CSV generation function ( unchanged )
        $generateCsv = function ( $users, $filenamePrefix ) {
            if ( $users->isEmpty() ) return null;
            $csv = Writer::createFromString( '' );
            $csv->setDelimiter( ',' );
            $csv->setEnclosure( '"' );
            $csv->setEscape( '\\' );
            $csv->insertOne( [ 'Sno', 'Email', 'Phone' ] );
            $csv->insertAll( $users->map( fn( $user ) => [ $user->id, $user->email, $user->phone ] )->toArray() );
            $filename = "{$filenamePrefix}_" . now()->format( 'Y-m-d_His' ) . '.csv';
            $filePath = storage_path( "app/{$filename}" );
            file_put_contents( $filePath, $csv->getContent() );
            return [ 'path' => $filePath, 'filename' => $filename ];
        } ;


        foreach ( $filters as $key => $filter ) {
            if ( optional($filterers)[$key] ) {
                $users = $filter[ 'query' ]()->get();
                if($csv){
                    if ( $attachment = $generateCsv( $users, $filter[ 'filename' ] ) ) {
                        $attachments[] = $attachment;
                    }
                }
                $usersList = array_merge($usersList, $users->toArray());
            }
        }

       return ['users' => $usersList, 'attachments' => $attachments];
    }
}
