<?php

namespace App\Models;

use App\Enums\AudienceGroupEnum;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Log;

class Voucher extends Model {
    protected $fillable = [
        'user_id',
        'investment_scheme_id',
        'code',
        'max_usage',
        'usage_per_person',
        'used_count',
        'discount_value',
        'audience'
    ];

    /**
    * The user this voucher is assigned to ( nullable, for personal vouchers ).
    */

    public function user(): BelongsTo {
        return $this->belongsTo( User::class, 'user_id'  );
    }

    public function owner() {
        return $this->belongsTo( User::class, 'user_id' );
    }

    /**
    * The investment scheme this voucher is linked to.
    */

    public function investmentScheme(): BelongsTo {
        return $this->belongsTo( InvestmentScheme::class, 'investment_scheme_id', 'id' );
    }

    /**
    * Users and how many times they've used this voucher.
    */

    public function usage(): HasMany {
        return $this->hasMany( UserVoucherUsage::class );
    }


    public function scopeGlobalAndUsableBy( $query, $userId ) {
        return $query->whereNull( 'user_id' )
        ->whereRaw( '(
            SELECT COALESCE(SUM(is_used), 0)
            FROM user_voucher_usages
            WHERE user_id = ? AND voucher_id = vouchers.id
        ) < COALESCE(usage_per_person, 999999)', [ $userId ] );
    }

    public function scopeForAudience($query, array|string $audiences)  {
        $audiences = is_array($audiences) ? $audiences : [$audiences];

        return $query->where(function ($q) use ($audiences) {
            $q->whereNull('audience')
              ->orWhere('audience', AudienceGroupEnum::All()->value)
              ->orWhereIn('audience', $audiences);
        });
    }

    public static function getUseableBy(User $user): Collection {
        $audience = $user->getAudienceGroups(); 
        return self::query()
            ->where(function ($query) use ($user) {
                // Global vouchers or those from user's upline
                $query->whereNull('user_id');

                if ($user->upline) {
                    $query->orWhere('user_id', $user->upline->id);
                }
            })
            ->where(function ($query) use ($user) {
                // Check if user hasn't exceeded personal usage limit
                $query->whereNull('usage_per_person')
                    ->orWhereRaw('
                        (
                            SELECT COALESCE(SUM(is_used), 0)
                            FROM user_voucher_usages
                            WHERE user_id = ? AND voucher_id = vouchers.id
                        ) < COALESCE(usage_per_person, 999999)
                    ', [$user->id]);
            })
            ->where(function ($query) use ($user) {
                // Exclude vouchers created by the user themselves
                $query->whereNull('user_id')
                    ->orWhere('user_id', '!=', $user->id);
            })
            ->forAudience($audience)
            ->with(['owner:id', 'investmentScheme'])
            ->get();
    }

}
