<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;

class Rental extends Model {
    public $fillable = [
        'company_id',
        'robot_id',
        'quantity',
        'amount_per_hour',
        'rent_duration_months',
        'revenue_generated',
        'last_revenue_processed_at',
        'next_revenue_at',
        'status'
    ];

    protected $casts = [
        'last_revenue_processed_at' => 'datetime',
        'next_revenue_at' => 'datetime'
    ];

    protected static function boot() {
        parent::boot();
        static::creating( function ( $model ) {
            if(!$model->last_revenue_processed_at){
                $model->last_revenue_processed_at = Carbon::now();
                $model->next_revenue_at =  $model->last_revenue_processed_at->copy()->addHour($model->robot->working_hours);
            }
        });
    }

    public function company() {
        return $this->belongsTo( Company::class );
    }

    public function robot() {
        return $this->belongsTo( Robot::class );
    }

    public function calculateRevenue(): float {
        // Base revenue per second per robot
        $baseRevenuePerRobotPerSecond = $this->robot->renting_cost_per_hour / 3600;

        // Total revenue per second for all rented robots
        $revenuePerSecond = $this->quantity * $baseRevenuePerRobotPerSecond;

        // Calculate intervals
        $lastPayoutTime = $this->last_revenue_processed_at;
        $currentTime = Carbon::now();

        // Total elapsed time in seconds
        $elapsedTimeInSeconds = abs($currentTime->diffInSeconds($lastPayoutTime));

        if ($elapsedTimeInSeconds > 0) {
            return $elapsedTimeInSeconds * $revenuePerSecond;
        }

        return 0.0; // If no time has passed
    }

}
