<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Enums\InterestPeriod;
use App\Enums\InterestType;
use App\Enums\InvestmentSchemeType;
use App\Enums\PlanDurationPeriod;
use App\Enums\StatusEnum;
use OwenIt\Auditing\Contracts\Auditable;
use App\Models\InvestmentMaturityBonusTier;

class InvestmentScheme extends Model implements Auditable {
    use HasFactory, \OwenIt\Auditing\Auditable;

    protected $fillable = [
        'scheme_name',
        'learn_more_link',
        'type',
        'sid',
        'short_name',
        'description',
        'min_invest_amount',
        'max_invest_amount',
        'interest_rate',
        'interest_type',
        'interest_period',
        'term_duration',
        'term_duration_period',
        'status',
        'compounding_bonus',
        'interest_plans',
        'risk_exposure_percentage',
    ];

    protected $casts = [
        'interest_type' => InterestType::class,
        'interest_period' => InterestPeriod::class,
        'term_duration_period' => PlanDurationPeriod::class,
        'type' => InvestmentSchemeType::class,
        'interest_plans' => 'array',
        'compounding_bonus' => 'decimal:4'
    ];

    protected $appends = [
        'total_investors',
        'maturity_bonus_tiers'
    ];

    /**
    * Relationship with the Investment model
    * A scheme can have many investments.
    */
    public function getTotalInvestorsAttribute(): int  {
        return User::whereHas('investments', function ($query) {
            $query->where('scheme_id', $this->id);
        })->count();
    }

    public function getMaturityBonusTiersAttribute() {
        $mbt = $this->maturityBonusTiers()->get();

        return $mbt->map(function($tier) {
            $tier->investment_scheme_name = $this->scheme_name;
            $tier->investment_scheme_sid = $this->sid;
            return $tier;
        });
    }

    public function investments() {
        return $this->hasMany( Investment::class, 'scheme_id' );
    }

    public function promos() {
        return $this->hasMany( Promo::class, 'scheme_id', 'id' );
    }

    public function scheme_active_promo() {
        return $this->hasOne( Promo::class, 'scheme_id', 'id' ) ->where( 'status', StatusEnum::ACTIVE )
        ->where( 'start_date', '<=', now() )
        ->where( 'end_date', '>=', now() );
    }

    public function vouchers() {
        return $this->hasMany(Voucher::class);
    }

    public function active_vouchers(){
        return $this->hasMany(Voucher::class)->where(function ($query) {
            // Placeholder, this will be scoped dynamically later
        });
    }

    public function maturityBonusTiers() {
         return $this->hasMany(InvestmentMaturityBonusTier::class, 'investment_scheme_id')
          ->orderBy('min_amount');
    }
}
