<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;
use OwenIt\Auditing\Contracts\Auditable;
use App\Models\InvestmentMaturityBonusTier;

class Investment extends Model implements Auditable {
    use HasFactory, \OwenIt\Auditing\Auditable;

    protected $fillable = [
        'transaction_id',
        'investment_end_date',
        'last_roi_date',
        'next_payout_time',
        'total_returns',
        'status',
        'scheme_id',
        'interest_plan',
        'amount_invested',
        'last_roi_amount',
        'last_roi_date',
        'applied_investment_tier_id'
    ];

    protected $casts = [
        'investment_end_date' => 'date',
        'last_roi_amount' => 'decimal:5',
        'last_roi_date' => 'date',
        'next_payout_time' => 'datetime',
        'total_returns' => 'decimal:5',
        'amount_invested' => 'decimal:5',
    ];

    /**
    * Relationship with the InvestmentScheme model
    */

    public function scheme() {
        return $this->belongsTo( InvestmentScheme::class, 'scheme_id' );
    }

    /**
    * Relationship with the Transaction model
    */

    public function transaction() {
        return $this->belongsTo( Transaction::class, 'transaction_id' );
    }

    /**
    * Custom relationship to get the User model through the Transaction model
    */

    public function user() {
        return $this->transaction ? $this->transaction->user : null;
    }

    public function getMaturityBonusPercentageAttribute(): float {
        $tier = $this->getActiveMaturityBonusTier();

        if ( ! $tier ) return 0;

        if ( $tier->max_amount === null || $this->amount_invested <= $tier->max_amount ) {
            return $tier->bonus_percentage;
        }

        return 0;
    }

    public function getMaturityBonusAmountAttribute(): float {
        return round( $this->amount_invested * $this->maturity_bonus_percentage / 100, 2 );
    }

    public function appliedBonusTier() {
        return $this->belongsTo( InvestmentMaturityBonusTier::class, 'applied_investment_tier_id' );
    }
}
