<?php

namespace App\Http\Middleware;

use App\Models\SystemSettings;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class RequireFresh2FA {
    public function handle( Request $request, Closure $next ) {
        try {
            $user = $request->user();

            // Is 2-FA globally enabled?
            if ( system_setting( SystemSettings::ENABLED_2FA, false ) && $user->hasEnabledTwoFactorAuthentication() ) {

                // User has never completed 2-FA
                if ( !$user?->two_factor_verified_at ) {
                    return response()->json( [
                        'message' => '2FA not verified. Please re-verify.'
                    ], 403 );
                }

                // Validity window ( seconds ) – default 12 h
                $validity = ( float ) config( 'auth.twofa_validity', 60 * 60 * 12 );

                // Seconds since last successful verification
                $secondsSince2FA = $user->two_factor_verified_at->diffInSeconds( now() );

                if ( $secondsSince2FA > $validity ) {
                    return response()->json( [
                        'message' => '2FA expired. Please re-verify.'
                    ], 403 );
                }
            }

            return $next( $request );

        } catch ( \Throwable $e ) {

            // 🚨  Detailed failure log
            Log::error( '2FA middleware failed', [
                'user_id' => $request->user()?->id,
                'error'   => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
            ] );

            return response()->json( [
                'message' => 'An unexpected error occurred while validating 2FA. Please try again.'
            ], 500 );
        }
    }
}
