<?php

namespace App\Helpers;

use Carbon\Carbon;
use App\Enums\InterestType;
use App\Enums\PlanDurationPeriod;
use App\Enums\InterestPeriod;
use App\Enums\InterestPlan;
use App\Enums\InvestmentSchemeType;
use App\Enums\PayoutIntervalEnumSec;
use App\Models\Investment;
use App\Models\InvestmentScheme;
use Illuminate\Support\Facades\Log;

class InvestmentCalculator {
    /**
    * Calculate investment details based on the investment instance.
    *
    * @param Investment $investment
    * @return array
    */
    public static function calculateInvestmentDetails( Investment $investment ): array {
        $scheme = $investment->scheme;
        // Calculate profit
        $profit = self::calculateProfit( $investment );
        // Calculate end date
        $startDate = $investment->created_at;
        $endDate = self::calculateEndDate( $startDate, $scheme );
        // Calculate next payout time
        $lastPayoutTime = $investment->updated_at;
        $nextPayoutTime = self::calculateNextPayoutTime( Carbon::now(), $scheme );
        return [
            'profit' => $profit,
            'end_date' => $endDate->toDateString(),
            'next_payout_time' => $nextPayoutTime,
        ];
    }

    public static function getBaseProfit( Investment $investment ): float {
        $scheme = $investment->scheme;

        if ( !$scheme?->interest_type ) return 0.0;

        $amount = ( float ) ( $investment->amount_invested ?? 0.0 );

        // Use bonus tier rate if available, otherwise scheme rate
        $rate =  $scheme->interest_rate;
        if($investment->appliedBonusTier?->interest_percentage_on_investment){
            $rate = ( float ) $investment->appliedBonusTier?->interest_percentage_on_investment;
        }

        return match ( $scheme->interest_type?->value ) {
            InterestType::PERCENTAGE  => $amount * ( $rate / 100 ),
            InterestType::FIXED => ( float ) $rate,
            default => 0.00
        }
        ;
    }

    /**
    * Calculate the profit based on the investment amount and scheme details.
    *
    * @param Investment $investment
    * @return float
    */
    public static function calculateProfit( Investment $investment ): float {
        $scheme = $investment->scheme;
        $interestPlan = $investment->interest_plan;
        $compoundingBonus = $scheme->compounding_bonus;
        $interestFrequency = self::getPayoutInterval( $scheme->interest_period );

        $profit = 0.0;
        $profitMultiplier = 1;
        $interest_frequency = ( float ) system_setting( 'interest_frequency_hours', config( 'system_settings.interest_frequency_hours', 0 ) );
        $payoutInterval = $interest_frequency > 0
        ? $interest_frequency * 3600
        : $interestFrequency;

        if ( $interest_frequency > 0 ) {
            $profitMultiplier = ( $payoutInterval / $interestFrequency );
        }

        $baseProfit = self::getBaseProfit( $investment );

        $currentTime = Carbon::now();
        $nextPayoutTime = $investment->next_payout_time;

        // Skip weekends for next payout time
        while ( $nextPayoutTime->isSaturday() || $nextPayoutTime->isSunday() ) {
            $nextPayoutTime->addDay();
        }

        // Process missed payouts
        while ( $currentTime >= $nextPayoutTime ) {
            // Add profit for the missed payout
            $profit += $baseProfit;
            // Adjust for compounding
            if ( $interestPlan === InterestPlan::COMPOUNDING ) {
                $profit += ( $baseProfit * ( $compoundingBonus / 100 ) );
            }

            // Move the next payout time forward
            $nextPayoutTime->addSeconds( $payoutInterval );

            // Skip weekends for updated next payout time
            while ( $nextPayoutTime->isSaturday() || $nextPayoutTime->isSunday() ) {
                $nextPayoutTime->addDay();
            }
        }

        // Update the investment's next payout time in the database
        // $investment->next_payout_time = $nextPayoutTime;

        // Apply the profit multiplier
        $P = round($profit * $profitMultiplier, 8);

        return $P;
    }

    /**
    * Get the payout interval in seconds based on the interest period.
    *
    * @param InterestPeriod $interestPeriod
    * @return int
    */

    public static function getPayoutInterval(InterestPeriod $interestPeriod): int {
        return match ($interestPeriod->value) {
            InterestPeriod::DAILY => PayoutIntervalEnumSec::DAILY,
            InterestPeriod::WEEKLY => PayoutIntervalEnumSec::WEEKLY,
            InterestPeriod::MONTHLY => PayoutIntervalEnumSec::MONTHLY,
            InterestPeriod::YEARLY => PayoutIntervalEnumSec::YEARLY,
            InterestPeriod::HOURLY => PayoutIntervalEnumSec::HOURLY,
            default => 0,
        };
    }


    public static function calculateEndDate(Carbon $startDate, InvestmentScheme $scheme): Carbon {
        $duration = (int) $scheme->term_duration;
        $period   = strtoupper($scheme->term_duration_period);

        if ($duration <= 0) {
            throw new \InvalidArgumentException("Invalid term duration: {$scheme->term_duration}");
        }

        $validPeriods = [
            PlanDurationPeriod::DAY,
            PlanDurationPeriod::WEEK,
            PlanDurationPeriod::MONTH,
            PlanDurationPeriod::YEAR,
            PlanDurationPeriod::HOUR,
        ];

        if (!in_array($period, $validPeriods, true)) {
            throw new \InvalidArgumentException("Invalid term duration period: {$scheme->term_duration_period}");
        }

        $date = $startDate->copy()->startOfDay();

        // Everything now counts only Monday–Friday
        return match ($period) {
            PlanDurationPeriod::DAY   => $date->addWeekdays($duration),           // 10 DAY  → 10 working days
            PlanDurationPeriod::WEEK  => $date->addWeekdays($duration * 5),        // 1 WEEK  → 5 working days
            PlanDurationPeriod::MONTH => static::addBusinessMonths($date, $duration), // 1 MONTH → ~20–22 working days
            PlanDurationPeriod::YEAR  => static::addBusinessMonths($date, $duration * 12), // 1 YEAR → ~250–252 working days
            PlanDurationPeriod::HOUR  => $date->addHours($duration),               // hours stay normal
            default                   => $date,
        };
    }

    /**
     * Add X business months (skip weekends)
     */
    private static function addBusinessMonths(Carbon $date, int $months): Carbon {
        $endDate = $date->copy()->addMonthsNoOverflow($months);
        $targetDayOfWeek = $date->dayOfWeek;
        if ($endDate->isSaturday()) {
            $endDate->addDays(2);
        } elseif ($endDate->isSunday()) {
            $endDate->addDay();
        }
        return $endDate;
    }


    /**
    * Calculate the next payout time based on the scheme's payout frequency.
        *
        * @param Carbon $lastPayoutTime
        * @param InvestmentScheme $scheme
        * @return Carbon
        */
        public static function calculateNextPayoutTime( Carbon $lastPayoutTime, InvestmentScheme $scheme ): Carbon {
            $interest_frequency = ( float ) system_setting( 'interest_frequency_hours', config( 'system_settings.interest_frequency_hours' ) );
            $nextPayoutTime = null;

            if ( $interest_frequency > 0 ) {
                $nextPayoutTime = $lastPayoutTime->copy()->addHours( $interest_frequency );
            } else {
                switch ( $scheme->interest_period ) {
                    case InterestPeriod::DAILY:
                    $nextPayoutTime = $lastPayoutTime->copy()->addDay();
                    break;
                    case InterestPeriod::WEEKLY:
                    $nextPayoutTime = $lastPayoutTime->copy()->addWeek();
                    break;
                    case InterestPeriod::MONTHLY:
                    $nextPayoutTime = $lastPayoutTime->copy()->addMonth();
                    break;
                    case InterestPeriod::YEARLY:
                    $nextPayoutTime = $lastPayoutTime->copy()->addYear();
                    break;
                    case InterestPeriod::HOURLY:
                    $nextPayoutTime = $lastPayoutTime->copy()->addHour();
                    break;
                }
            }

            // Adjust next payout time if it falls on a weekend
            while ( $nextPayoutTime->isSaturday() || $nextPayoutTime->isSunday() ) {
                $nextPayoutTime->addDay();
            }

            $specificHour = 14;
            $nextPayoutTime->setTime( $specificHour, 0, 0 );

            return $nextPayoutTime;
        }

        /**
        * Convert the term duration into years for compounding calculations.
        *
        * @param InvestmentScheme $scheme
        * @return float
        */
        protected static function getTermInYears( InvestmentScheme $scheme ): float {
            switch ( $scheme->term_duration_period ) {
                case PlanDurationPeriod::DAY:
                return $scheme->term_duration / 365;
                case PlanDurationPeriod::WEEK:
                return $scheme->term_duration / 52;
                case PlanDurationPeriod::MONTH:
                return $scheme->term_duration / 12;
                case PlanDurationPeriod::YEAR:
                return $scheme->term_duration;
                case PlanDurationPeriod::HOUR:
                return $scheme->term_duration / ( 24 * 365 );
                default:
                return 0;
            }
        }
    }
