<?php

namespace App;

use App\Enums\StatusEnum;
use App\Enums\TransactionType;
use App\Enums\UserWalletType;
use App\Helpers\InvestmentCalculator;
use App\Models\Investment;
use App\Models\InvestmentMaturityBonusTier;
use App\Models\Transaction;
use App\Services\PromoService;
use App\Services\PromoUseService;
use App\Services\VoucherService;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

trait HasTransactions {
    /**
    * Deposit amount into the user's wallet.
     *
     * @param float $amountUsd
     * @param float $amountCrypto
     * @param mixed $methodInfo
     * @param string $status
     * @return \App\Models\Transaction
     * @throws \Exception
     */
    public function deposit(float $amountUsd, float $amountCrypto, mixed $methodInfo, string $status = StatusEnum::PENDING): Transaction {
        return $this->createTransaction(
            amountUsd: $amountUsd,
            amountCrypto: $amountCrypto,
            methodInfo:$methodInfo[ 'payment_method' ] ?? $methodInfo,
            type: TransactionType::Deposit,
            status: $status
        );
    }

       /**
    * Deposit amount into the user's wallet.
    *
    * @param float $amountUsd
    * @param float $amountCrypto
    * @param mixed $methodInfo
    * @param string $status
    * @return \App\Models\Transaction
    * @throws \Exception
    */

    public function topUpInvestment( float $amountUsd, float $amountCrypto, mixed $methodInfo, string $status = StatusEnum::PENDING ): Transaction {
        return $this->createTransaction(
            amountUsd: $amountUsd,
            amountCrypto: $amountCrypto,
            methodInfo:$methodInfo[ 'payment_method' ] ?? $methodInfo,
            type: TransactionType::Topupinvestment,
            status: $status,
            expire_time: $status == StatusEnum::PENDING ? Carbon::now()->addDay(  ):NULL
        );
    }


    public function getWalletBalance( string $walletType ): float {
        switch ($walletType) {
            case UserWalletType::WalletBalance:
                return $this->wallet_balance ?? 0.0;
            case UserWalletType::ReferralCommissionBalance:
                return $this->referral_commission_balance ?? 0.0;
            case UserWalletType::PremiumInterestBalance:
                return $this->premium_interest_balance ?? 0.0;
            default:
                throw new \Exception("Unsupported wallet type: {$walletType}");
        }
    }

    public function withdraw(
        float $amountUsd,
        float $amountCrypto,
        mixed $methodInfo,
        string $walletFrom = UserWalletType::WalletBalance,
        string $status = StatusEnum::PENDING
    ): Transaction {
        if (!UserWalletType::hasValue($walletFrom)) {
            throw new \Exception("Invalid wallet type: {$walletFrom}");
        }

        if($walletFrom == UserWalletType::ReferralCommissionBalance){
            $user = $this;
            $userAffiliatePackage = $user->affiliatePackageOrDefault();
            $referralsCount = $user->referrals()->count();
            $minRequired = $userAffiliatePackage['benefits']['min_referrals_required_for_cashout'] ?? 1;
            if ($referralsCount < $minRequired) {
                throw new \Exception("You need at least {$minRequired} referrals before you can request a cashout.");
            }
        }

        $walletBalance = $this->getWalletBalance($walletFrom);
        if ($walletBalance < $amountUsd) {
            throw new \Exception("Insufficient funds in {$walletFrom}. Available: {$walletBalance} USD, Requested: {$amountUsd} USD");
        }
        $description = match ($walletFrom) {
            UserWalletType::WalletBalance => "Withdrawal of {$amountUsd} USD from main wallet balance.\n",
            UserWalletType::ReferralCommissionBalance => "Withdrawal of {$amountUsd} USD from referral commission balance.\n",
            UserWalletType::PremiumInterestBalance => "Withdrawal of {$amountUsd} USD from premium interest balance.\n",
            default => "Withdrawal of {$amountUsd} USD from account\n"
        };

        $methodInfo = is_array($methodInfo) ? $methodInfo : [];
        $methodInfo['payment_method'] = $methodInfo['payment_method'] ?? $methodInfo;
        $methodInfo['description'] = $description;
        $methodInfo['wallet_from'] = $walletFrom;

        // Create and return transaction
        return $this->createTransaction(
            amountUsd: $amountUsd,
            amountCrypto: $amountCrypto,
            methodInfo: $methodInfo,
            type: TransactionType::Withdrawal,
            status: $status,
        );
    }


    /**
     * Transfer referral bonus from the user's referral balance into their wallet.
        *
        * @param array|string $methodInfo  Information about the payment method.
        * @return \App\Models\Transaction
        * @throws \Exception
        */

        public function transferReferralBonusToWallet( array|string $methodInfo = [] ): Transaction {
            $transferAmount = $this->referral_commission_balance;

            // Ensure method info is structured
            if ( is_string( $methodInfo ) ) {
                $methodInfo = [ 'payment_method' => $methodInfo ];
            }

            $methodInfo[ 'description' ] = "You requested to transfer your referral bonus of {$transferAmount} USD to your wallet balance.";

            return $this->createTransaction(
                amountUsd: $transferAmount,
                amountCrypto: 0.00,
                methodInfo: $methodInfo,
                type: TransactionType::Transfer,
                status: StatusEnum::PENDING
            );
        }

        /**
        * Withdraw amount from the user's wallet.
     *
     * @param float $amountUsd
     * @param mixed $methodInfo
     * @param string $status
     * @return \App\Models\Transaction
     * @throws \Exception
     */
    public function earned(float $amountUsd,  mixed $methodInfo, string $status = StatusEnum::APPROVED): Transaction {
        return $this->createTransaction(
            amountUsd: $amountUsd,
            amountCrypto: 0,
            methodInfo: $methodInfo,
            type: TransactionType::Earnings,
            status: $status
        );
    }

    /**
     * Add bonus amount to the user's wallet.
        *
        * @param float $amountUsd
        * @param mixed $methodInfo
        * @param string $status
        * @return \App\Models\Transaction
        * @throws \Exception
        */

        public function addBonus( float $amountUsd, mixed $methodInfo, string $status = StatusEnum::APPROVED ): Transaction {
            return $this->createTransaction(
                amountUsd: $amountUsd,
                amountCrypto: 0,
                methodInfo:$methodInfo[ 'payment_method' ] ?? $methodInfo,
                type: TransactionType::Bonus,
                status: $status
            );
        }

        /**
        * Add bonus amount to the user's wallet.
    *
    * @param float $amountUsd
    * @param mixed $methodInfo
    * @param string $status
    * @return \App\Models\Transaction
    * @throws \Exception
    */

    public function addCommission( float $amountUsd, mixed $methodInfo, string $status = StatusEnum::APPROVED ): Transaction {
        return $this->createTransaction(
            amountUsd: $amountUsd,
            amountCrypto: 0,
            methodInfo:$methodInfo[ 'payment_method' ] ?? $methodInfo,
            type: TransactionType::Commission,
            status: $status
        );
    }

    public function createDepositChargeTransaction(float $amountUsd, mixed $methodInfo, string $status = StatusEnum::PENDING): Transaction {
        DB::beginTransaction();
        try {
            // Check if user has invested plans
            if ($this->invested_plans->isEmpty()) {
                Log::warning("User ID: {$this->id} has no invested plans, cannot charge fee.");
                throw new \Exception("User has no invested plans.");
            }

            if ($this->hasNoInvestment()) {
                Log::warning("User ID: {$this->id} has no investments, cannot charge fee.");
                throw new \Exception("No investments found for user.");
            }

            $transaction = $this->createTransaction(
                $amountUsd,
                0,
                $methodInfo,
                TransactionType::FeeDeposit,
                $status
            );

            DB::commit();
            return $transaction;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Error creating deposit charge transactions for user ID: {$this->id}: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Add bonus amount to the user's wallet.
        *
        * @param float $amountUsd
        * @param Investment $investment
        * @param mixed $methodInfo
        * @param string $status
        * @return \App\Models\Transaction
        * @throws \Exception
        */

        public function addBonusToInvestment( float $amountUsd, Investment $investment, mixed $methodInfo, string $status = StatusEnum::APPROVED ) {
            DB::beginTransaction();
            try {
                if ( $investment ) {
                    if ( $status === StatusEnum::APPROVED ) {
                        $investment->increment( 'amount_invested', $amountUsd );
                    }
                    $transaction = $this->createTransaction(
                        amountUsd: $amountUsd,
                        amountCrypto: 0,
                        methodInfo: $methodInfo[ 'payment_method' ] ?? $methodInfo,
                        type: TransactionType::Bonus,
                        status: $status
                    );
                    DB::commit();
                    return $transaction;
                }
            } catch ( \Exception $e ) {
                DB::rollBack();
                throw $e;
            }
        }

        /**
        * Create an investment deposit for the user.
        *
        * @param float $amountUsd
        * @param float $amountCrypto
        * @param mixed $methodInfo
        * @param string $status
        * @return \App\Models\Transaction
        * @throws \Exception
        */

        public function createInvestmentDeposit( array $investmentData, string $status = StatusEnum::PENDING ): Transaction {
            $methodInfo = [
                'description' => 'Investment deposit',
                ...$investmentData[ 'payment_method' ],
            ];

            // Create the transaction
            $transaction = $this->createTransaction(
                amountUsd: $investmentData[ 'amount_usd' ],
                amountCrypto: $investmentData[ 'amount_crypto' ],
                methodInfo: $methodInfo,
                type: TransactionType::Investment,
                status: $status,
                expire_time: $status == StatusEnum::PENDING ? Carbon::now()->addDay(  ):NULL
            );

            $schemeId = $investmentData['scheme_id'];
            $amount = $investmentData[ 'amount_usd' ];

            $bonusTier = InvestmentMaturityBonusTier::active()
                ->currentlyRunning()
                ->where(function ($q) use ($schemeId) {
                    $q->where('investment_scheme_id', $schemeId);
                    // ->orWhereNull('investment_scheme_id');
                })
                ->where(function ($qu) use ($amount){
                    $qu->where('min_amount', '<=', $amount)
                    ->where('max_amount', '>=', $amount);
                })->first();

            // Create the investment linked to the transaction
            $investment = $this->investments()->create( [
                ...$investmentData,
                'transaction_id' => $transaction->id,
                'investment_end_date' => Carbon::now(),
                'interest_plan' => $investmentData[ 'interest_plan' ],
                'status' => $status === StatusEnum::APPROVED ? StatusEnum::ACTIVE : $status,
                'amount_invested' => $investmentData[ 'amount_usd' ],
                'applied_investment_tier_id' => $bonusTier?->id,
            ] );

            if ( $voucherCode = optional( $investmentData )[ 'voucher_code' ] ) {
                $actualAmount = VoucherService::calculateDiscount( $voucherCode, $transaction->user, $transaction->amount_usd, $investment->scheme->id );
                $methodInfo[ 'description' ] = $methodInfo[ 'description' ]. ". Applied vouch code $voucherCode. -$".$actualAmount;
                $transaction->update( [
                    'amount_usd' => $transaction->amount_usd - $actualAmount,
                    'method_info' => $methodInfo
                ] );
                $vouchedUsed = VoucherService::apply( $voucherCode, $transaction->user, $investment->scheme->id, $transaction->id );
            }

            // Attach the investment data to the transaction ( for return purposes )
            $transaction->setRelation( 'investment', $investment );
            return $transaction;
        }

        /**
        * Create an investment deposit for the user.
        *
        * @param float $amountUsd
        * @param float $amountCrypto
        * @param mixed $methodInfo
        * @param string $status
        * @return \App\Models\Transaction
        * @throws \Exception
        */

        public function createReInvestmentDeposit( array $investmentData, string $status = StatusEnum::WAITING ): Transaction {
            $methodInfo = [
                'description' => 'Investment deposit',
                ...$investmentData[ 'payment_method' ],
            ];

            // Create the transaction
            $transaction = $this->createTransaction(
                amountUsd: $investmentData[ 'amount_usd' ],
                amountCrypto: $investmentData[ 'amount_crypto' ],
                methodInfo: $methodInfo,
                type: TransactionType::ReInvestment,
                status: $status,
                expire_time:NULL
            );

            // Create the investment linked to the transaction
            $investment = $this->investments()->create( [
                ...$investmentData,
                'transaction_id' => $transaction->id,
                'investment_end_date' => Carbon::now(), // You may want to calculate an actual end date here
                'interest_plan' => $investmentData[ 'interest_plan' ],
                'status' => $status === StatusEnum::APPROVED ? StatusEnum::ACTIVE : $status,
                'amount_invested' => $investmentData[ 'amount_usd' ],
            ] );

            // Attach the investment data to the transaction ( for return purposes )
            $transaction->setRelation( 'investment', $investment );
            return $transaction;
        }

        /**
        * Create a new transaction for the user.
        *
        * @param float $amountUsd
        * @param float $amountCrypto
        * @param mixed $methodInfo
        * @param string $type
        * @param string $status
        * @return \App\Models\Transaction
        * @throws \Exception
        */

        public function createTransaction(
            float $amountUsd,
            float $amountCrypto,
            mixed $methodInfo,
            string $type,
            string $status,
            mixed $expire_time = null
        ): Transaction {

            switch ( $type ) {
                case TransactionType::Commission:
                if ( $status === StatusEnum::APPROVED ) {
                    $this->increment( 'referral_commission_balance', $amountUsd );
                }
                break;
                case TransactionType::Bonus:
                if ( $status === StatusEnum::APPROVED ) {
                    if ( !optional( $methodInfo )[ 'credit_investment' ] ) {
                        Log::info( 'not credit_investment' );
                        $this->increment( 'wallet_balance',  $amountUsd );
                    }
                }
                break;
                case TransactionType::Transfer:
                if ( $status === StatusEnum::APPROVED ) {
                    $this->increment( 'wallet_balance',  $amountUsd );
                }
                $this->decrement( 'referral_commission_balance', $amountUsd );
                break;
                case TransactionType::Withdrawal:
                    $updated = $this->where('id', $this->id)
                        ->where($methodInfo['wallet_from'], '>=', $amountUsd)
                        ->decrement($methodInfo['wallet_from'], $amountUsd);

                    if (!$updated) {
                        throw new \Exception('Insufficient balance');
                    }
                break;
            }

            return $this->transactions()->create( [
                'amount_usd' => $amountUsd,
                'amount_crypto' => $amountCrypto,
                'method_info' => $methodInfo,
                'type' => $type,
                'status' => $status,
                'expire_time' => $expire_time
            ] );
        }
    public function cancelTransaction(Transaction $transaction): Transaction {
        if ($transaction->status !== StatusEnum::PENDING) {
            return $transaction;
        }

        DB::beginTransaction();
        try {
            if ($transaction->type == TransactionType::Withdrawal) {
                $walletFrom = $transaction->method_info['wallet_from'];

                // Validate wallet type
                if (!UserWalletType::hasValue($walletFrom)) {
                    throw new \Exception("Invalid wallet type: {$walletFrom}");
                }

                // Increment the appropriate wallet balance
                $transaction->user->increment($walletFrom, $transaction->amount_usd);

                // Update transaction description
                $transaction->method_info = [
                    ...$transaction->method_info,
                    'description' => $transaction->method_info['description'] . " Withdrawal of {$transaction->amount_usd} USD from {$walletFrom} was cancelled."
                ];
            }

            if ($transaction->type == TransactionType::Transfer) {
                $transaction->user->increment('referral_commission_balance', $transaction->amount_usd);

                // Update transaction description
                $transaction->method_info = [
                    ...$transaction->method_info,
                    'description' => $transaction->method_info['description'] . " Transfer of {$transaction->amount_usd} USD was cancelled."
                ];
            }

            $transaction->update(['status' => StatusEnum::CANCELLED]);

            if ($transaction->investment) {
                $transaction->investment->update(['status' => StatusEnum::CANCELLED]);
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error cancelling transaction: ' . $e->getMessage());
            throw $e;
        }

        return $transaction;
    }

        public function setTransactionWaiting( Transaction $transaction, string $transaction_hash ): Transaction {
            if ( $transaction->status !== StatusEnum::PENDING ) {
                return $transaction;
            }
            $transaction->method_info = [ ...$transaction->method_info, 'transaction_hash' => $transaction_hash ];
            $transaction->status =  StatusEnum::WAITING;
            $transaction->save();
            if ( $transaction->investment ) {
                if($transaction->type !== TransactionType::Topupinvestment)
                   $transaction->investment->update( [ 'status'=> StatusEnum::WAITING ] );
            }
            return $transaction;
        }

    public function declineTransaction(Transaction $transaction): Transaction {
        if ($transaction->status === StatusEnum::PENDING || $transaction->status === StatusEnum::WAITING) {
            DB::beginTransaction();
            try {
                if ($transaction->type == TransactionType::Withdrawal) {
                    $walletFrom = $transaction->method_info['wallet_from'];
                    if (!UserWalletType::hasValue($walletFrom)) {
                        throw new \Exception("Invalid wallet type: {$walletFrom}");
                    }
                    $transaction->user->increment($walletFrom, $transaction->amount_usd);
                    $transaction->method_info = [
                        ...$transaction->method_info,
                        'description' => $transaction->method_info['description'] . " Withdrawal of {$transaction->amount_usd} USD from {$walletFrom} was declined."
                    ];
                }

                if ($transaction->type == TransactionType::Transfer) {
                    $transaction->user->increment('referral_commission_balance', $transaction->amount_usd);
                    $transaction->method_info = [
                        ...$transaction->method_info,
                        'description' => $transaction->method_info['description'] . " Transfer of {$transaction->amount_usd} USD was declined."
                    ];
                }

                $transaction->update(['status' => StatusEnum::DECLINED]);
                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                Log::error('Error declining transaction: ' . $e->getMessage());
                throw $e;
            }
        }

        return $transaction;
    }
    public function approveTransaction(Transaction $transaction): Transaction {
        if ($transaction->status === StatusEnum::APPROVED) {
            return $transaction;
        }

        DB::beginTransaction();
        try {
            $additionalData = null;
            $type = $transaction->type;

            switch ($type) {
                case TransactionType::Investment:
                    $investment = $transaction->investment;
                    $scheme = $investment->scheme;
                    $investment->status = StatusEnum::ACTIVE;
                    $investment->investment_end_date = InvestmentCalculator::calculateEndDate(Carbon::now(), $scheme);
                    $investment->next_payout_time = InvestmentCalculator::calculateNextPayoutTime(Carbon::now(), $scheme);
                    $promo = $scheme->scheme_active_promo;

                    if (optional($transaction->method_info)['is_added_by_admin'] !== true && $promo) {
                        if ($promo->start_amount <= $investment->amount_invested) {
                            $prmoBonusApplied = PromoService::calculateBonusAmount($promo, $investment->amount_invested);
                            $investment->amount_invested += $prmoBonusApplied;

                            $additionalData = [
                                'description' => sprintf(
                                    '🎉 Congratulations! You have received a bonus of $%s through the "%s" promo. Your total investment is now $%s. Thank you for investing with us!',
                                    number_format($prmoBonusApplied, 2),
                                    $promo->title,
                                    number_format($investment->amount_invested, 2)
                                )
                            ];
                            PromoUseService::applyPromoToTransaction($transaction->user->id, $transaction->id, $promo->id, $additionalData);
                        }
                    }

                    if ($transaction->userVoucherUsage) {
                        $transaction->userVoucherUsage->update(['is_used' => true]);
                        $transaction->userVoucherUsage->voucher->increment('used_count');
                    }

                    $investment->save();
                    break;

                case TransactionType::Commission:
                    $transaction->user->increment('referral_commission_balance', $transaction->amount_usd);
                    break;

                case TransactionType::Bonus:
                    if (optional($transaction->method_info)['credit_investment'] !== true) {
                        $transaction->user->increment('wallet_balance', $transaction->amount_usd);
                    }
                    break;

                case TransactionType::InvestedCapitalWithdrawal:
                case TransactionType::Transfer:
                    $transaction->user->increment('wallet_balance', $transaction->amount_usd);
                    break;

                case TransactionType::FeeDeposit:
                    // Handle FeeDeposit case, if necessary
                    break;

                case TransactionType::Withdrawal:
                    $walletFrom = $transaction->method_info['wallet_from'] ?? UserWalletType::WalletBalance;

                    // Validate wallet type

                    if (!UserWalletType::hasValue($walletFrom)) {
                        throw new \Exception("Invalid wallet type: {$walletFrom}");
                    }

                    // Check sufficient balance
                    // $currentBalance = $this->getWalletBalance($walletFrom);
                    // if ($currentBalance < $transaction->amount_usd) {
                    //     throw new \Exception("Insufficient funds in {$walletFrom}. Available: {$currentBalance} USD, Requested: {$transaction->amount_usd} USD");
                    // }

                    // Decrement the appropriate wallet balance
                    // $transaction->user->decrement($walletFrom, $transaction->amount_usd);

                    $additionalData = [
                        'description' => "Withdrawal of {$transaction->amount_usd} USD from {$walletFrom} has been approved."
                    ];
                    break;

                case TransactionType::Topupinvestment:
                    $investment = Investment::where('id', optional($transaction->method_info)['investment_id'])->firstOrFail();
                    $investment->increment('amount_invested', $transaction->amount_usd);
                    break;

                case TransactionType::ReInvestment:
                    $investment = $transaction->investment;
                    $scheme = $investment->scheme;
                    $investment->status = StatusEnum::ACTIVE;
                    $investment->investment_end_date = InvestmentCalculator::calculateEndDate(Carbon::now(), $scheme);
                    $investment->next_payout_time = InvestmentCalculator::calculateNextPayoutTime(Carbon::now(), $scheme);
                    $investment->save();
                    break;

                default:
                    throw new \Exception('Invalid transaction type');
            }

            if ($additionalData) {
                $transaction->method_info = [
                    ...$transaction->method_info,
                    'description' => $transaction->method_info['description'] . ' ' . optional($additionalData)['description'] ?? ''
                ];
            }

            $transaction->status = StatusEnum::APPROVED;
            $transaction->save();
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error approving transaction: ' . $e->getMessage());
            throw $e;
        }

        return $transaction;
    }

        /**
        * Get the transactions associated with the user.
        *
        * @return \Illuminate\Database\Eloquent\Relations\HasMany
        */

        public function transactions() {
            return $this->hasMany( Transaction::class );
        }

        /**
        * Get all transactions for the user.
        *
        * @return \Illuminate\Database\Eloquent\Collection
        */

        public function getAllTransactions() {
            return $this->transactions()->get();
        }
    }
