<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Enums\InterestPlan;
use App\Enums\InvestmentSchemeType;
use App\Enums\StatusEnum;
use App\Enums\TransactionType;
use App\Helpers\InvestmentCalculator;
use App\Models\Investment;
use App\Services\AffiliateService;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class ProcessMaturedInvestments extends Command {
    /**
    * The name and signature of the console command.
    *
    * @var string
    */
    protected $signature = 'app:process-matured-investments';

    /**
    * The console command description.
    *
    * @var string
    */
    protected $description = 'Processes investments that are due for payout based on their next payout time and status.';

    /**
    * Execute the console command.
    */
    public function handle() {
        $chunkSize = 100;
        $this->info("Starting to process matured investments in chunks of $chunkSize...");
        Investment::whereIn('status', [StatusEnum::ACTIVE])
        ->where('next_payout_time', '<=', now())
        ->chunk($chunkSize, function ($investments) {
                $this->info("Processing chunk with {$investments->count()} investments...");
                foreach ($investments as $investment) {
                    try {
                        $this->processPayout($investment);
                    } catch (\Exception $e) {
                        $this->error("Failed to process investment ID {$investment->id}: " . $e->getMessage());
                    }
                }
            });
        $this->info("Completed processing of matured investments.");
    }

    /**
    * Dispatch a job to handle the payout for a matured investment.
    *
    * @param \App\Models\Investment $investment
    */
    private function processPayout(Investment $investment) {
        try {
            if (!$investment) {
                $this->error('Investment instance is not available for processing.');
                return;
            }
            $details = InvestmentCalculator::calculateInvestmentDetails($investment);
            $profit = $details['profit'];
            $nextPayoutTime = $details['next_payout_time'];
            if ($profit <= 0) {
                $this->info("No profit calculated for investment. 'investment_id' => $investment->id, profit => $profit");
                return;
            }
            DB::transaction(function () use ($investment, $profit, $nextPayoutTime, $details) {
                self::processInvestment($investment, $profit, $nextPayoutTime, Carbon::parse($details['end_date']));
            });
        } catch (\Exception $e) {
              Log::error('Error in ProcessMaturedInvestments->processPayout: ' . $e->getMessage(), [
                'investment_id' => $investment->id,
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]);
        }
    }


    /**
     * Process the investment based on calculated profit and end date.
     *
     * @param Investment $investment
     * @param float $profit
     * @param Carbon|null $nextPayoutTime
     * @param Carbon $endDate
     */
    protected static function processInvestment(Investment $investment, float $profit, ?Carbon $nextPayoutTime, Carbon $endDate): void
    {
        $investment->total_returns += $profit;

        // if (!$nextPayoutTime || Carbon::now()->greaterThanOrEqualTo($endDate)) {
        //     self::completeInvestment($investment, $profit);
        // } else {
            self::updateNextPayout($investment, $profit, $nextPayoutTime);
        // }

        $investment->update(['last_roi_date' => now()]);
    }

    /**
     * Complete the investment and handle payouts for compounding and non-compounding plans.
     *
     * @param Investment $investment
     * @param float $profit
     */
    public static function completeInvestment(Investment $investment, float $profit): void {
        $investment->status = StatusEnum::INACTIVE;
        $investment->next_payout_time = null;
        $investment->save();
        $amount_invested = $investment->amount_invested ;

        if (self::isCompounding($investment)) {
            $total = $investment->total_returns;
            //  + $amount_invested;
            if($investment->scheme->type->value === InvestmentSchemeType::Premium){
                // $investment->user()->increment('premium_interest_balance', $profit);
            }else{
               // $investment->user()->increment('wallet_balance', $total);
            }
            AffiliateService::distributeProfitCommission($investment->transaction, $investment->total_returns);
        } else {
            $total = $profit + $amount_invested;
            // $investment->user()->increment('wallet_balance', $total);
            // AffiliateService::distributeProfitCommission($investment->transaction,  $profit);
        }
        // $investment->user()->earned($profit, [
        //     'description' => "An interest of $$profit has been credited to you as the investment has also concluded.",
        //     'amount_invested' => $investment->amount_invested,
        //     'initial_capital' => $investment->transaction->amount_usd,
        //     'investment_id' => $investment->id,
        //     'scheme_name' => $investment->scheme->scheme_name,
        // ]);
        $methodInfo = [
            'description' => "Investment account closure for amount $".$total.'.',
            'amount_invested' => $investment->amount_invested,
            'initial_capital' => $investment->transaction->amount_usd,
            'investment_id' => $investment->id,
            'scheme_name' => $investment->scheme->scheme_name,
            ];
        $investment->user()->createTransaction(
            $total,
            0,
            $methodInfo,
            TransactionType::InvestedCapitalWithdrawal,
            StatusEnum::WAITING
        );
    }

    /**
     * Update the next payout time for the investment and handle the payout for non-compounding plans.
     *
     * @param Investment $investment
     * @param float $profit
     * @param Carbon $nextPayoutTime
    */
    protected static function updateNextPayout(Investment $investment, float $profit, Carbon $nextPayoutTime): void{
        $investment->next_payout_time = $nextPayoutTime;

        if (!self::isCompounding($investment)) {
            if($investment->scheme->type->value === InvestmentSchemeType::Premium){
                $investment->user()->increment('premium_interest_balance', $profit);
            } else {
                $investment->user()->increment('wallet_balance', $profit);
                AffiliateService::distributeProfitCommission($investment->transaction,  $profit);
            }
        }  else {
                if($investment->amount_invested <= 0){
                    $profit_am = $profit +  ($investment->transaction->amount_usd + $investment->total_returns);
                } else {
                    $profit_am  = $profit;
                }
                $investment->amount_invested += $profit_am;
        }

        if ($investment->interest_plan === InterestPlan::FLEXIBLE) {
            $description = "A flexible interest of $$profit credited to your ".strtolower($investment->scheme->scheme_name)." plan.";
        } elseif ($investment->interest_plan === InterestPlan::COMPOUNDING) {
            $description = "A compounded interest of $$profit added to your ".strtolower($investment->scheme->scheme_name)." plan.";
        } else {
            $description = strtolower($investment->interest_plan)." interest of $$profit added to ".strtolower($investment->scheme->scheme_name).".";
        }

        $investment->last_roi_amount = $profit;
        $investment->save();
        $investment->user()->earned($profit, [
            'description' => $description,
            'amount_invested' => $investment->amount_invested,
            'initial_capital' => $investment->transaction->amount_usd,
            'investment_id' => $investment->id,
            'scheme_name' => $investment->scheme->scheme_name,
        ]);
    }

    /**
     * Check if the investment's interest plan is compounding.
     *
     * @param Investment $investment
     * @return bool
     */
    protected static function isCompounding(Investment $investment): bool
    {
        return Str::lower($investment->interest_plan) === Str::lower(InterestPlan::COMPOUNDING);
    }
}
