<?php

namespace App\Services;

use App\Models\User;
use App\Models\Voucher;
use App\Models\UserVoucherUsage;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class VoucherService
{
    public static function create(array $data): Voucher
    {
        return Voucher::create($data);
    }

    public static function update(Voucher $voucher, array $data): Voucher
    {
        $voucher->update($data);
        return $voucher;
    }

    public static function reset(Voucher $voucher): void
    {
        $voucher->update(['used_count' => 0]);

        UserVoucherUsage::where('voucher_id', $voucher->id)->update([
            'usage_count' => 0,
        ]);
    }

    public static function apply(string $code, User $user, int $investmentSchemeId, int $transactionId): Voucher
    {
        $voucher = self::getAllowedVoucher($user, $code, $investmentSchemeId);

        if (!$voucher) {
            throw ValidationException::withMessages([
                'voucher' => 'Invalid or non-applicable voucher in apply.',
            ]);
        }

        $totalUsageCount = UserVoucherUsage::where('user_id', $user->id)->where('is_used', '>', 0)->where('voucher_id', $voucher->id)->count();
        self::validateUsage($voucher, $totalUsageCount);
        $userUsage = UserVoucherUsage::firstOrCreate([
            'user_id'    => $user->id,
            'voucher_id' => $voucher->id,
            'transaction_id' => $transactionId
        ]);
        // DB::transaction(function () use ($voucher, $userUsage) {
        //     $voucher->increment('used_count');
        //     $userUsage->increment('usage_count');
        // });

        return $voucher;
    }

    public static function calculateDiscount(string $code, User $user, float $investmentAmount, int $investmentSchemeId): float
    {
        $voucher = self::getAllowedVoucher($user, $code, $investmentSchemeId);
        if (!$voucher) {
            throw ValidationException::withMessages([
                'voucher' => 'Invalid or non-applicable voucher in calculate Discount',
            ]);
        }

        $totalUsageCount = UserVoucherUsage::where('user_id', $user->id)->where('is_used', '>', 0)->where('voucher_id', $voucher->id)->count();
        if($totalUsageCount){
            self::validateUsage($voucher,$totalUsageCount);
        }

        return ($investmentAmount * $voucher->discount_value) / 100;
    }

    public static function validateUsage(Voucher $voucher,int $totalUsageCount): void
    {
        if ($voucher->max_usage > 0 && $voucher->used_count >= $voucher->max_usage) {
            throw ValidationException::withMessages([
                'voucher' => 'This voucher has reached its global usage limit.',
            ]);
        }

        if (
            !is_null($voucher->usage_per_person) &&
            $totalUsageCount >= $voucher->usage_per_person
        ) {
            throw ValidationException::withMessages([
                'voucher' => 'You have already used this voucher the maximum number of times.',
            ]);
        }
    }

    public static function remainingUsesGlobal(Voucher $voucher): int
    {
        return max(0, $voucher->max_usage - $voucher->used_count);
    }

    public static function delete(Voucher $voucher): void
    {
        $voucher->delete();
    }

    public static function listAvailable(User $user, int $investmentSchemeId)  {
        $uplinerIds = [$user->upline?->id];

        return Voucher::where('investment_scheme_id', $investmentSchemeId)
            ->where(function ($q) use ($user, $uplinerIds) {
                $q->whereNull('user_id')
                  ->orWhereIn('user_id', $uplinerIds);
            })
            ->forAudience($user->getAudienceGroups())
            ->get();
    }


    public static function getAllowedVoucher(User $user, string $code, int $investmentSchemeId): ?Voucher {
        $uplinerIds = [$user->upline?->id];
        return Voucher::where('code', $code)
            ->where('investment_scheme_id', $investmentSchemeId)
            ->where(function ($q) use ($uplinerIds) {
                $q->whereNull('user_id')
                  ->orWhereIn('user_id', $uplinerIds);
            })
            ->forAudience($user->getAudienceGroups())
            ->first();
    }

}
