<?php

namespace App\Services;

use App\Enums\NotificationType;
use App\Enums\TransactionType;
use App\Enums\StatusEnum;
use App\Jobs\EmailJob;
use App\Models\User;
use App\Models\Transaction;
use App\Models\SystemSettings;
use Illuminate\Support\Facades\Log;

class NotificationService {
    private static function sendNotification(
        string $type,
        ?string $chatId = null,
        bool $sendEmail = true,
        ?Transaction $transaction = null,
        ?User $user = null
    ): void {
        $method = match ( $type ) {
            NotificationType::TransactionCreated => 'sendTransactionCreatedNotification',
            NotificationType::TransactionUpdated => 'sendTransactionUpdatedNotification',
            NotificationType::Generic => 'sendGenericNotification',
            NotificationType::Login => 'sendLoginNotification',
            NotificationType::AdminTransaction => 'sendAdminTransactionNotification',
            NotificationType::AdminUserJoin => 'sendAdminUserJoinNotification',
            NotificationType::ReferralJoin => 'sendReferralJoinNotification',
        }
        ;

        $transactionLink = $transaction
        ? system_setting( SystemSettings::SITE_URL, 'https://default.site' ) . "/finances?id={$transaction->id}&type={$transaction->type}"
        : null;

        $finalMessage = match ( $type ) {
            NotificationType::TransactionCreated => $transaction
            ? self::generateTransactionCreatedMessage( $transaction, $transactionLink )
            : 'New transaction created',
            NotificationType::TransactionUpdated => $transaction
            ? self::generateTransactionUpdatedMessage( $transaction, $transactionLink )
            : 'Transaction updated',
            NotificationType::Login => $user
            ? "{$user->name} logged in at " . now()->toDateTimeString()
            : 'User login detected',
            NotificationType::Generic => 'System notification',
            NotificationType::AdminTransaction => $transaction
            ? self::generateAdminTransactionMessage( $transaction, $transactionLink )
            : 'New transaction processed',
            NotificationType::AdminUserJoin => $user
            ? "New user {$user->name} (ID: {$user->id}) joined"
            : 'New user joined',
            NotificationType::ReferralJoin => $user
            ? "Your referral {$user->name} has joined!"
            : 'New referral joined',
        }
        ;

        // if ( $transaction && ( str_starts_with( $type, 'transaction' ) || $type === NotificationType::AdminTransaction ) ) {
        //     $finalMessage = "<b>Transaction #{$transaction->id}</b>: " . $finalMessage;
        // } else {
        //     $finalMessage = htmlspecialchars( $finalMessage );
        // }

        if ( $sendEmail ) {
            EmailJob::dispatch( [
                'transaction' => $transaction,
                'type' => $type,
                'message' => $finalMessage,
                'user'=> $user
            ] )->afterCommit();
        }
    }

    private static function generateTransactionCreatedMessage( Transaction $transaction, ?string $transactionLink ): string {
        $type = TransactionType::getDescription( $transaction->type );
        $amount = $transaction->amount_usd;
        $date = $transaction->created_at->toDateTimeString();

        return "A new {$type} transaction of \${$amount} has been created on {$date}. " .
        "<a href='{$transactionLink}'>View Transaction</a>";
    }

    private static function generateTransactionUpdatedMessage( Transaction $transaction, ?string $transactionLink ): string {
        $type = TransactionType::getDescription( $transaction->type );
        $status = StatusEnum::getDescription( $transaction->status );
        $amount = $transaction->amount_usd;
        $date = $transaction->updated_at->toDateTimeString();

        return "The status of your {$type} transaction of \${$amount} has been updated to '{$status}' on {$date}. " .
        "<a href='{$transactionLink}'>View Transaction</a>";
    }

    private static function generateAdminTransactionMessage( Transaction $transaction, ?string $transactionLink ): string {
        $type = TransactionType::getDescription( $transaction->type );
        $status = StatusEnum::getDescription( $transaction->status );
        $amount = $transaction->amount_usd;
        $userId = $transaction->user_id;
        $userName = $transaction->user->first_name . ' ' . $transaction->user->last_name;

        return "A new {$type} transaction of \${$amount} (ID: {$transaction->id}) by User ID: {$userId}, {$userName} has been processed with status '{$status}'. " .
        "<a href='{$transactionLink}'>View Transaction</a>";
    }

    public static function notifyTransactionCreated( string $chatId, Transaction $transaction, bool $sendEmail = true ): void {
        self::sendNotification( NotificationType::TransactionCreated, $chatId, $sendEmail, $transaction );
    }

    public static function notifyTransactionCreatedForUser( User $user, Transaction $transaction, bool $sendEmail = true ): void {
        self::sendNotification( NotificationType::TransactionCreated, $user->telegram_chat_id, $sendEmail, $transaction, $user );
    }

    public static function notifyTransactionUpdated( string $chatId, Transaction $transaction, bool $sendEmail = true ): void {
        self::sendNotification( NotificationType::TransactionUpdated, $chatId, $sendEmail, $transaction );
    }

    public static function notifyTransactionUpdatedForUser( User $user, Transaction $transaction, bool $sendEmail = true ): void {
        self::sendNotification( NotificationType::TransactionUpdated, $user->telegram_chat_id, $sendEmail, $transaction, $user );
    }

    public static function notifyLogin( string $chatId, bool $sendEmail = false ): void {
        self::sendNotification( NotificationType::Login, $chatId, $sendEmail );
    }

    public static function notifyLoginForUser( User $user, bool $sendEmail = false ): void {
        self::sendNotification( NotificationType::Login, $user->telegram_chat_id, $sendEmail, null, $user );
    }

    public static function notifyGeneric( string $chatId, bool $sendEmail = false ): void {
        self::sendNotification( NotificationType::Generic, $chatId, $sendEmail );
    }

    public static function notifyGenericForUser( User $user, bool $sendEmail = false ): void {
        self::sendNotification( NotificationType::Generic, $user->telegram_chat_id, $sendEmail, null, $user );
    }

    public static function notifyAdminNewTransaction( Transaction $transaction, bool $sendEmail = false ): void {
        try {
            $adminChatId = env( 'DNX_LOG_CHAT_ID', '' );
            self::sendNotification( NotificationType::AdminTransaction, $adminChatId, $sendEmail, $transaction );
        } catch ( \Throwable $th ) {
            Log::channel( 'dnx' )->error( $th->getMessage() );
        }
    }

    public static function notifyAdminUserJoin( string $adminChatId, User $user, bool $sendEmail = false ): void {
        self::sendNotification( NotificationType::AdminUserJoin, $adminChatId, $sendEmail, null, $user );
        Log::channel( 'dnx' )->info( "Admin notified of user join: {$user->first_name}" );
    }

    public static function notifyUserReferralJoin( User $user, bool $sendEmail = false ): void {
        self::sendNotification( NotificationType::ReferralJoin, $user->telegram_chat_id, $sendEmail, null, $user );
        Log::channel( 'dnx' )->info( "User notified of referral join: {$user->first_name}" );
    }
}
