<?php

namespace App\Models;

use App\Enums\AffiliatePackageType;
use App\Enums\StatusEnum;
use App\AffiliateBenefits;
use App\Enums\PayoutIntervalEnumSec;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AffiliatePackage extends Model {
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'benefits',
        'is_system_default',
        'required_referrals_to_join',
        'status',
        'user_id'
    ];

    protected $casts = [
        'benefits' => 'array',
    ];

   protected $appends = [
    'max_referral_commission_level',
    'initial_deposit_bonus_percentage',
    'has_joined_programme',
    'joined_programme_name',
    'is_default',
    'user_affiliate_packages_count'
];

    public function scopeActive($query) {
        return $query->whereHas('userAffiliatePackages', function ($query) {
            $query->where('status', 'active');
        });
    }

    public function calculateDepositCommissionForLevel(int $currentDepositCount, int|float $amount, bool $isDirect = true): float {
        if ($isDirect) {
            $commission = $this->benefits['direct_referral_bonus']["deposit_$currentDepositCount"] ?? 0;
            if(!$commission){
                $commission = $this->benefits['direct_referral_bonus']['subsequent_deposits'] ?? 0;
            }
        } else {
            $commission = $this->benefits['indirect_referral_bonus']["deposit_$currentDepositCount"] ?? 0;
            if(!$commission){
                $commission = $this->benefits['indirect_referral_bonus']['subsequent_deposits'] ?? 0;
            }
        }
        $commissionPercentage = $commission / 100;
        return round(max(0, $commissionPercentage * $amount), 8);
    }

    public function calculateProfitCommissionForLevel(float $amount, bool $isDirect = true): float {
        // Determine the commission percentage based on whether it's direct or indirect
        $commissionPercentage = $isDirect
            ? ($this->benefits['weekly_profit_from_direct']['percentage'] ?? 0) / 100
            : ($this->benefits['monthly_profit_from_indirects']['percentage'] ?? 0) / 100;

        // Calculate the total commission for the period (weekly or monthly)
        $totalCommission = $commissionPercentage * $amount;

        // Determine the number of hours in the period
        $hoursInPeriod = $isDirect
            ? PayoutIntervalEnumSec::WEEKLY
            : PayoutIntervalEnumSec::MONTHLY;

        // Calculate the hourly commission
        $comm = $totalCommission / $hoursInPeriod;

        // Ensure the commission is not negative and round to 8 decimal places
        return round(max(0, $comm), 8);
    }

    public function isActiveBasedOnCriteria(int|float $amount): bool {
        if ($this->criteria_type == AffiliatePackageType::Investment) {
            return $amount >= $this->criteria_value;
            // Check if investment amount meets the criteria
        }

        if ($this->criteria_type == AffiliatePackageType::Referrals) {
            return $amount >= $this->criteria_value;
            // Check if profit count meets the criteria
        }

        return false;
    }

    public function calculateDirectProfitCommission(float $amount): float {
        return $amount * ($this->direct_profit_commission / 100);
    }

    public function calculateIndirectProfitCommission(float $amount): float {
        return $amount * ($this->indirect_profit_commission / 100);
    }

    public function totalUsers(): int {
        return $this->userAffiliatePackages()->count();
    }

    public function userAffiliatePackages() {
        return $this->hasMany(UserAffiliatePackage::class);
    }

    public function setDefaultValues()  {
        $b =  (new AffiliateBenefits);
        $this->title = 'System Default';
        $this->description = '';
        $this->benefits = $b->benefits;
        $this->is_default = true;
        $this->is_system_default = true;
        return $this;
    }

    public function meetsCriteriaToJoin(User $user): bool {
        $investRefsSum = $user->referredUsers()->whereHas('transactions.investment', function ($query) {
            $query->whereIn('status', [StatusEnum::ACTIVE, StatusEnum::COMPLETED]);
        })->count();
        return $investRefsSum >= $this->required_referrals_to_join;
    }

    public function meetsCriteriaToEarnIndirect(User $user): bool {
        $investRefsSum = $user->referredUsers()->whereHas('transactions.investment', function ($query) {
           $query->whereIn('status', [StatusEnum::ACTIVE, StatusEnum::COMPLETED]);
        })->count();

        return $investRefsSum >= $this->benefits['required_indirect_referrals'] ?? 1000 ;
    }

    public static function getDefaultAffiliatePackage() {
        $affPackage = AffiliatePackage::where('is_system_default', true)->first();
        if(!$affPackage)
        $affPackage =( new AffiliatePackage)->setDefaultValues();
        return $affPackage;
    }

    public function getMaxReferralCommissionLevelAttribute(): float  {
        return (float)system_setting('max_referral_commission_level', 20);
    }

    public function getUserAffiliatePackagesCountAttribute(): float {
        $query = User::query();
        if ($this->is_system_default) {
            $query->whereDoesntHave('activeAffiliatePackage', function ($query) {});
        } else {
            $query->whereHas('activeAffiliatePackage', function ($query) {});
        }
        return $query->count();
    }

    public function getInitialDepositBonusPercentageAttribute(): float  {
        return $this->direct_deposit_benefit ?? (float)system_setting('initial_deposit_bonus_percentage', 0);
    }

    public function getIsDefaultAttribute(): bool  {
        $user =  request()->user('sanctum') ;
        if ($user) {
            return $this->userAffiliatePackages()
                ->where('user_id', $user->id)
                ->exists();
        }
        return false;;
    }

    public function getHasJoinedProgrammeAttribute(): bool {
        $user =  request()->user('sanctum') ;
        if ($user) {
            return $this->userAffiliatePackages()
                ->where('user_id', $user->id)
                ->exists();
        }
        return false;
    }

    public function getJoinedProgrammeNameAttribute(): ?string {
        if ($this->hasJoinedProgramme) {
            return $this->title;
        }
        return null;
    }
}
