<?php

namespace App\Jobs;

use App\Enums\NotificationType;
use App\Enums\StatusEnum;
use App\Enums\TransactionType;
use App\Mail\DepositApprovedEmail;
use App\Mail\DepositComplimentEmial;
use App\Mail\ReferralPaidEmail;
use App\Mail\SendAdminNotification;
use App\Mail\TransactionCreated;
use App\Mail\TransactionUpdated;
use App\Mail\UserLoggedIn;
use App\Mail\UserRegistered;
use App\Mail\WithdrawalApprovedEmail;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class EmailJob implements ShouldQueue {
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $uniqueFor = 600;

    /**
    * Create a new job instance.
    *
    * @param array $emailData
    *   Example structure:
    *   [
        *       'type' => 'login', // or 'registration', 'reset_password', etc.
        *       'user' => $user,
        *       'extra_data' => [ ... ],
        * ]
        */

        public function __construct( public array $emailData ) {
            $this->emailData = $emailData;
        }

        // Generate a unique identifier for the job

        public function uniqueId() {
            return $this->emailData[ 'type' ] . '-' . optional( $this->emailData )[ 'user' ]->id ?? optional( $this->emailData )[ 'transaction' ]?->user?->id;
        }

        /**
        * Execute the job.
        *
        * @return void
        */

        public function handle() {

            /** @var \App\Models\User $user */
            $user = optional( $this->emailData )[ 'user' ];
            if ( !$user ) return ;

            switch ( $this->emailData[ 'type' ] ) {

                case NotificationType::DepositComplimentEmail:
                Mail::to( $user->email )->send( new DepositComplimentEmial( $user ) );
                break;

                case NotificationType::Login:
                // Send login email
                Mail::to( $user->email )->send( new UserLoggedIn( $user ) );
                break;

                case NotificationType::Registration:
                // Send registration email
                Mail::to( $user->email )->send( new UserRegistered( $user ) );
                break;

                case NotificationType::TransactionCreated:
                $transaction = $this->emailData[ 'transaction' ];
                if ( $transaction->type == TransactionType::Commission && $transaction->status == StatusEnum::APPROVED ) {
                    $referred = User::where( 'id', optional( $transaction->method_info )[ 'from_user_id' ] )->first();
                    if ( !$referred ) return;
                    Mail::to( $transaction->user->email )
                    ->send( new ReferralPaidEmail( $transaction, $referred ) );
                    return;
                }
                Mail::to( $transaction->user->email )->send( new TransactionCreated( $transaction ) );
                break;

                case NotificationType::TransactionUpdated:
                $transaction = $this->emailData[ 'transaction' ];

                // Send transaction-updated email
                if ( $transaction->status === 'PENDING' || $transaction->status === 'WAITING' ) {
                    if ( system_setting( 'admin_email' ) ) {
                        Mail::to( system_setting( 'admin_email' ) )->send(
                            new SendAdminNotification( $transaction->user, $transaction )
                        );
                    }
                    if ( system_setting( 'support_email' ) ) {
                        Mail::to( system_setting( 'support_email' ) )->send(
                            new SendAdminNotification( $transaction->user, $transaction )
                        );
                    }
                }

                if ( $transaction->type == TransactionType::Withdrawal && $transaction->status == StatusEnum::APPROVED ) {
                    Mail::to( $transaction->user->email )->send( new WithdrawalApprovedEmail( $transaction ) );
                    Log::info($transaction->user->email);
                    return;
                }

                if ( $transaction->type == TransactionType::Investment && $transaction->status == StatusEnum::APPROVED ) {
                    Mail::to( $transaction->user->email )
                    ->send( new DepositApprovedEmail( $transaction ) );
                    return;
                }

                if ( $transaction->type == TransactionType::Commission && $transaction->status == StatusEnum::APPROVED ) {
                    if ( $transaction->amount_usd <= 0 ) return;
                    if ( !str_contains( optional( $transaction->method_info )[ 'commission_type' ], 'deposit' ) ) return;
                    $referred = User::where( 'id', optional( $transaction->method_info )[ 'from_user_id' ] )->first();
                    if ( !$referred ) return;
                    Mail::to( $transaction->user->email )->send( new ReferralPaidEmail( $transaction, $referred ) );
                    return;
                }

                // Mail::to( $transaction->user->email )->send( new TransactionUpdated( $transaction ) );
                break;

                // case 'reset_password':
                // // Send password reset email
                // Mail::to( $this->emailData[ 'user' ]->email )->send( new \App\Mail\PasswordReset( $this->emailData[ 'extra_data' ] ) );
                // break;

                default:
                break;
            }
        }
    }
