<?php

namespace App\Http\Requests;

use App\Enums\InterestPlan;
use App\Enums\StatusEnum;
use App\Models\Investment;
use App\Models\InvestmentScheme;
use App\Models\SystemSettings;
use App\Models\Transaction;
use Illuminate\Foundation\Http\FormRequest;

class InvestmentRequest extends FormRequest {
    /**
    * Determine if the user is authorized to make this request.
    */

    public function authorize(): bool {
        return true;
    }

    /**
    * Get the validation rules that apply to the request.
    *
    * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
    */

    public function rules(): array {
        $scheme = InvestmentScheme::findOrFail( request( 'scheme_id' ) );
        $minAmount = $scheme->min_invest_amount;

        // Apply voucher discount first if present
        if ( $voucherCode = request( 'voucher_code' ) ) {
            $voucher = \App\Models\Voucher::where( 'code', $voucherCode )
            ->where( 'investment_scheme_id', $scheme->id )
            ->first();

            if ( $voucher ) {
                $discountedMin = $minAmount - ( ( $minAmount * $voucher->discount_value ) / 100 );
                $minAmount = max( 0, $discountedMin );
            }
        }

        // Handle TOPUP case
        if ( ( $tid = request( 'tid' ) ) && \strtoupper( request( 'type' ) ) === 'TOPUP' ) {
            $transaction = Transaction::where( 'id', $tid )->firstOrFail();
            $minAmount = ( \system_setting( SystemSettings::MIN_TOPUP_PERCENTAGE, 0 ) / 100 ) * $minAmount;
            $validatables = [];
        } else {
            $validatables = [
                'interest_plan' => [ 'nullable', 'in:' . implode( ',', InterestPlan::asArray() ) ]
            ];
        }

        // Base rules
        $validatables = array_merge( $validatables, [
            'amount_usd' => [ 'required', 'numeric', 'min:' . $minAmount, 'max:' . $scheme->max_invest_amount ],
            'amount_crypto' => [ 'required', 'numeric' ],
            'scheme_id' => [ 'required', 'exists:investment_schemes,id', 'integer' ],
            'payment_method' => [ 'required', 'array' ],
            'voucher_code' => [ 'sometimes', 'string' ],
            'description' => [ 'sometimes', 'string' ],
        ] );

        return $validatables;
    }

    /**
    * Customize the validation error messages.
    *
    * @return array<string, string>
    */

    public function messages(): array {
        return [
            'scheme_id.required' => 'The scheme ID is required.',
            'scheme_id.exists' => 'The selected scheme ID does not exist.',
            'interest_plan.in' => 'The selected interest plan is invalid.',
            'amount_usd.required' => 'Please enter investment amount.',
            'payment_method.required' => 'Please choose a payment method.',
        ];
    }
}
