<?php

namespace App\Http\Controllers\Api;

use App\Enums\AudienceGroupEnum;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Voucher;
use App\Services\VoucherService;

class VoucherController extends Controller {

    public function calculateDiscount( Request $request ) {
        $request->validate( [
            'code' => 'required|string',
            'investment_scheme_id' => 'required|integer',
        ] );

        $user = $request->user();

        try {
            $discount = VoucherService::calculateDiscount(
                $request->code,
                $user,
                1.0,
                $request->investment_scheme_id
            );

            return $this->success( [
                'discount_percent' => $discount,
            ], 'Discount calculated successfully.' );
        } catch ( \Exception $e ) {
            return $this->error( $e->getMessage(), 400 );
        }
    }

    public function available( Request $request ) {
        $request->validate( [
            'investment_scheme_id' => 'required|integer',
        ] );

        $user = $request->user();

        try {
            $vouchers = VoucherService::listAvailable( $user, $request->investment_scheme_id );

            return $this->success( [ 'available_vouchers' => $vouchers, ] );
        } catch ( \Exception $e ) {
            return $this->error( $e->getMessage(), 400 );
        }
    }

    // Admin only endpoints ( protect via middleware )

    public function createVoucher( Request $request ) {
        $request->validate( [
            'code' => 'required|string|unique:vouchers,code',
            'investment_scheme_id' => 'required|integer|exists:investment_schemes,id',
            'discount_value' => 'required|numeric|min:0|max:100',
            'max_usage' => 'nullable|integer|min:0|gte:usage_per_person',
            'usage_per_person' => 'nullable|integer|min:0',
            'user_id' => 'nullable|exists:users,id',
            'audience' => 'sometimes|in:'. implode( ',', AudienceGroupEnum::asArray() )
        ] );

        try {
            $voucher = VoucherService::create( $request->all() );

            return $this->success( [
                'voucher' => $voucher,
            ], 'Voucher created successfully.' );
        } catch ( \Exception $e ) {
            return $this->error( $e->getMessage(), 400 );
        }
    }

    public function update( Request $request, Voucher $voucher ) {
        $data = $request->validate( [
            'discount_value' => 'sometimes|numeric|min:0|max:100',
            'max_usage' => 'nullable|integer|min:0|gte:usage_per_person',
            'usage_per_person' => 'sometimes|integer|min:0',
            'audience' => 'sometimes|in:'. implode( ',', AudienceGroupEnum::asArray() )
        ] );

        try {
            $updated = VoucherService::update( $voucher, $data );

            return $this->success( [
                'voucher' => $updated,
            ], 'Voucher updated successfully.' );
        } catch ( \Exception $e ) {
            return $this->error( $e->getMessage(), 400 );
        }
    }

    public function destroy( Voucher $voucher ) {
        try {
            VoucherService::delete( $voucher );

            return $this->success( [], 'Voucher deleted successfully.' );
        } catch ( \Exception $e ) {
            return $this->error( $e->getMessage(), 400 );
        }
    }

    public function reset( Voucher $voucher ) {
        try {
            VoucherService::reset( $voucher );

            return $this->success( [], 'Voucher usage reset successfully.' );
        } catch ( \Exception $e ) {
            return $this->error( $e->getMessage(), 400 );
        }
    }
}
