<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\PromotionalMaterial;
use Illuminate\Http\Request;
use App\Enums\MediaType;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class PromotionalMaterialController extends Controller {

    public function index() {
        $materials = PromotionalMaterial::all();
        $user = request()->user();
        $materials = $materials->map(function ($item) use ($user){
            return [
                ...$item->toArray(),
                'meta_link' => url('api/'). "ref?mid=".$item->id."&ref=".$user->referral_code
            ];
        });

        return $this->success( $materials);
    }

    public function socialMediaPreviewMaterial(Request $request) {
        $request->validate([
            'mid' => 'required|integer',
        ]);

        // Fetch the promotional material using the 'mid'
        $promotionalMaterial = PromotionalMaterial::find($request->query('mid'));

        // Check if the promotional material exists
        if (!$promotionalMaterial) {
            abort(404, 'Promotional material not found.');
        }

        // Pass the metadata to the Blade view
        return view('social-preview-material', [
            'title' => $promotionalMaterial->title,
            'description' => $promotionalMaterial->description,
            'media_url' => $promotionalMaterial->media_url,
            'media_type' => $promotionalMaterial->media_type,
            'link' => rtrim(system_setting('site_url'), '/') . '/auth?mid=' . $promotionalMaterial->id . '&ref=' . urlencode($request->query('ref')),
        ]);
    }

    public function show( Request $request, $id ) {
        $promotionalMaterial = PromotionalMaterial::find( $id );
        if ( !$promotionalMaterial ) {
            return $this->error( '', 404 );
        }
        return $this->success( $promotionalMaterial );
    }

    public function store(Request $request) {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'media_file' => 'required|file|mimes:jpg,jpeg,png,mp4,webp,gif',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors()->first(), 422, $validator->errors());
        }

        DB::beginTransaction();

        try {
            $file = $request->file('media_file');
            $extension = $file->getClientOriginalExtension();
            $randomName = Str::random(10) . '.' . $extension;

            // Move the file to public/dia/
            $destinationPath = public_path('dia');
            $mediaType = $this->determineMediaType($file);
            $file->move($destinationPath, $randomName);

            $mediaUrl = 'dia/' . $randomName; // relative public URL

            $promotionalMaterial = PromotionalMaterial::create([
                'title' => $request->title,
                'description' => $request->description,
                'media_url' => asset($mediaUrl),
                'media_type' => $mediaType,
            ]);

            DB::commit();

            return $this->success($promotionalMaterial, 'Upload was successful', 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to upload promotional material: ' . $e->getMessage());
            return $this->error('Upload failed: ' . $e->getMessage(), 500);
        }
    }


    public function update( Request $request, $id ) {
        $promotionalMaterial = PromotionalMaterial::find( $id );
        if ( !$promotionalMaterial ) {
            return $this->error( 'Promotional Material not found', 404 );
        }

        $validator = Validator::make( $request->all(), [
            'title' => 'sometimes|required|string|max:255',
            'description' => 'sometimes|required|string',
            'media_file' => 'sometimes|required|file|mimes:jpg,jpeg,png,mp4',
        ] );

        if ( $validator->fails() ) {
            return $this->error( $validator->errors()->first(), 422, $validator->errors() );
        }

        // Start a transaction
        DB::beginTransaction();

        try {
            if ( $request->hasFile( 'media_file' ) ) {
                // Delete the old file from storage
                $oldMediaPath = public_path( $promotionalMaterial->media_url );
                if ( file_exists( $oldMediaPath ) ) {
                    unlink( $oldMediaPath );
                }

                // Generate a random filename with the original file extension
                $extension = $request->file( 'media_file' )->getClientOriginalExtension();
                $randomName = Str::random( 10 ) . '.' . $extension;

                // Store the new file in the storage app public directory
                $mediaPath = $request->file( 'media_file' )->storeAs( 'dia', $randomName, 'public' );
                $mediaUrl = 'storage/app/' . $mediaPath;
                $mediaType = $this->determineMediaType( $request->file( 'media_file' ) );

                $promotionalMaterial->update( [
                    'media_url' => $mediaUrl,
                    'media_type' => $mediaType,
                ] );
            }

            $promotionalMaterial->update( $request->only( 'title', 'description' ) );

            // Commit the transaction
            DB::commit();

            return $this->success( $promotionalMaterial, 'Promotional Material updated successfully' );
        } catch ( \Exception $e ) {
            // Rollback the transaction
            DB::rollBack();
            Log::error( 'Failed to update promotional material: ' . $e->getMessage() );
            return $this->error( 'Update failed: ' . $e->getMessage(), 500 );
        }
    }

    public function destroy( $id ) {
        $promotionalMaterial = PromotionalMaterial::find( $id );
        if ( !$promotionalMaterial ) {
            return $this->error( 'Promotional Material not found', 404 );
        }

        // Start a transaction
        DB::beginTransaction();

        try {
            // Delete the file from storage
            $mediaPath = public_path( $promotionalMaterial->media_url );
            if ( file_exists( $mediaPath ) ) {
                unlink( $mediaPath );
            }

            $promotionalMaterial->delete();

            // Commit the transaction
            DB::commit();

            return $this->success( [], 'Promotional Material deleted successfully' );
        } catch ( \Exception $e ) {
            // Rollback the transaction
            DB::rollBack();
            Log::error( 'Failed to delete promotional material: ' . $e->getMessage() );
            return $this->error( 'Deletion failed: ' . $e->getMessage(), 500 );
        }
    }

    private function determineMediaType( $file ) {
        $mimeType = $file->getMimeType();

        if ( strpos( $mimeType, 'image/' ) === 0 ) {
            return MediaType::IMAGE;
        } elseif ( strpos( $mimeType, 'video/' ) === 0 ) {
            return MediaType::VIDEO;
        }

        throw new \InvalidArgumentException( 'Unsupported media type.' );
    }
}
