<?php

namespace App\Http\Controllers\Api\Auth;

use App\Enums\PinVerificationType;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\AuthLoginRequest;
use App\Http\Requests\Api\AuthRegisterRequest;
use App\Http\Resources\Api\AccountResource;
use App\Models\Referral;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Observers\UserObserver;
use BenSampo\Enum\Rules\EnumValue;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller {
    /**
    * Register a new user.
    */

    public function register( AuthRegisterRequest $authRegisterRequest ) {
        $validated = $authRegisterRequest->validated();
        DB::beginTransaction();
        try {
            $user = new User();
            $user->email = $validated[ 'email' ];
            $user->first_name = $validated[ 'first_name' ];
            $user->last_name = $validated[ 'last_name' ];
            $user->password = Hash::make( $validated[ 'password' ] );
            $user->phone = $authRegisterRequest->input( 'phone' );
            $user->country = $authRegisterRequest->input( 'country' );
            $user->referral_code = Referral::makeReferralCode();
            $user->save();

            if ( $authRegisterRequest->has( 'referee_code' ) &&  trim( $validated[ 'referee_code' ] ) ) {
                $referee = User::where( 'referral_code', $validated[ 'referee_code' ] )->first();
                if ( $referee ) {
                    $referral = new Referral();
                    $referral->downline_id = $user->id;
                    $referral->referee_id = $referee->id;
                    $referral->save();
                } else {
                    return $this->error( 'Invalid referee code provided.' );
                }
            }

            $user->access_token = $user->createApiToken();
            DB::commit();
            $account = new AccountResource( $user );
            return $this->success( $account, 'Account created successfully', 201 );
        } catch ( \Exception $e ) {
            DB::rollBack();
            Log::info( $e->getMessage() );
            return $this->error( 'Failed to create account. Please try again.', 500 );
        }
    }

    public function reset_password( Request $request ) {
        $validator = Validator::make( $request->all(), [
            'type' => [ 'required', 'string', new EnumValue( PinVerificationType::class, true ) ],
            'email' => [ 'required', 'exists:users,email', 'email' ],
            'pin' => [ 'required', ],
            'password' => [ 'required', 'string', 'max:255' ]
        ] );
        if ( $validator->fails() ) {
            return $this->error( $validator->errors()->first(), 400, $validator->errors() );
        }
        $user = User::where( 'email', $request->input( 'email' ) )->first();
        $pinVerification = $user->pin()
        ->where( 'expires_at', '>', now() )
        ->latest( 'created_at' )
        ->first();
        if ( !$pinVerification || !$pinVerification->verified( $request->input( 'pin' ) ) ) {
            return $this->error( 'Invalid Pin' );
        }
        $user->password = Hash::make( $request[ 'password' ] );
        $user->two_factor_verified_at = null;
        $user->save();
        return $this->success( [], 'Password reset successfully' );
    }

    /**
    * Login a user and create a token.
    */

    public function login( AuthLoginRequest $authLoginRequest ) {
        $validated  = $authLoginRequest->validated();
        $user = User::where( 'email', $validated [ 'email' ] )->first();

        if ( !$user ) {
            return $this->error( 'The provided email is incorrect.' );
        }

        if ( !Hash::check( $validated [ 'password' ], $user->password ) ) {
            return $this->error( 'The provided password is incorrect.' );
        }

        $user->two_factor_verified_at = null;
        $user->save();

        $user->tokens()->delete();

        $user->access_token = $user->createApiToken();
        $account = new AccountResource( $user );
        ( new UserObserver )->loggedIn( $user );
        return $this->success( $account, 'Login successfully' );
    }

    /**
    * Logout the authenticated user and revoke the token.
    */

    public function logout( Request $request ) {
        $user = $request->user();
        $user->currentAccessToken()->delete();
        $user->two_factor_verified_at = null;
        $user->save();
        return $this->success( 'Logged out successfully' );
    }
}
