<?php

namespace App\Http\Controllers\Api\Admin;

use App\Enums\AudienceGroupEnum;
use App\Enums\FeeVariantEnum;
use App\Enums\InterestPlan;
use App\Enums\PromoType;
use App\Enums\StatusEnum;
use App\Enums\TransactionType;
use App\Http\Controllers\Controller;
use App\Jobs\BroadcastJob;
use App\Mail\CsvExportMail;
use App\Models\InvestmentMaturityBonusTier;
use App\Models\InvestmentScheme;
use App\Models\Promo;
use App\Models\SystemSettings;
use App\Models\Transaction;
use App\Models\User;
use App\Services\AccountService;
use App\Services\AdminService;
use App\Services\PromoService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class AdminController extends Controller {

    public function admin_dashboard() {
        $adminDashboard = AdminService::dashboard();
        return $this->success( $adminDashboard, );
    }

    public function admin_get_users( Request $request ) {
        $filters = $request->all();
        $users = AdminService::users( $filters, 10000 );
        return  $this->success( $users );
    }

    public function admin_get_user( int $user_id ) {
        $user = User::findOrFail( $user_id );
        $user[ 'finances' ] = AccountService::finances( $user );
        $user[ 'referrals' ] = AccountService::referrals( $user );
        $user[ 'dashboard' ] = AccountService::dashboard( $user );
        $user[ 'affiliate_package' ] = $user->affiliatePackageOrDefault();
        return  $this->success( $user );
    }

    public function admin_get_schemes( Request $request ) {
        $schemes = AdminService::schemes();
        return  $this->success( $schemes );
    }

    public function admin_get_transactions( Request $request ) {
        $filters = $request->all();
        $transactions = AdminService::transactions( $filters );
        return $this->success( $transactions );
    }

    public function admin_update_settings( Request $request ) {
        $data = $request->all();
        $result = AdminService::updateSettings( $data );
        return $this->success( $result, 'Settings updated 👍.' );
    }

    public function admin_get_payment_methods() {
        $paymentOptions = AdminService::paymentMethods();
        return $this->success( $paymentOptions );
    }

    public  function admin_update_transaction( Request $request, $id ) {

        $request->validate( [
            'status' => 'required|in:'.implode( ',', StatusEnum::asArray() )
        ], [] );

        $status = $request[ 'status' ];

        try {
            $transaction = Transaction::findOrFail( $id );
            if ( StatusEnum::APPROVED === $status ) {
                $transaction = $transaction->user->approveTransaction( $transaction );
            } elseif ( StatusEnum::DECLINED === $status ) {
                $transaction = $transaction->user->declineTransaction( $transaction );
            }
            return $this->success( $transaction, 'Transaction has been '.$status.'.' );
        } catch( \Throwable $th ) {
            $errors = $th->getMessage() ;
            Log::info( $errors );
            return $this->error( $errors, 500, $errors );
        }
    }

    public function admin_update_user_account( Request $request, $id ) {
        $request->validate( [
            'status' => 'sometimes|in:' . implode( ',', StatusEnum::asArray() ),
            'role' => 'sometimes|in:' . implode( ',', [ 'ADMIN', 'USER' ] ),
            'first_name' => 'sometimes|string|max:255',
            'last_name' => 'sometimes|string|max:255',
            'bio' => 'sometimes|string|nullable',
        ] );
        try {
            $user = User::findOrFail( $id );
            $fillable = $user->getFillable();
            $excludedFields = [ 'password', 'wallet_balance', 'email' ];
            $fieldsToUpdate = array_diff( $fillable, $excludedFields );
            $user->fill( $request->only( $fieldsToUpdate ) );
            $user->save();
            return $this->success( $user, 'User has been updated successfully.' );
        } catch ( \Throwable $th ) {
            Log::error( 'Error in AdminController -> update_user_account: ' . $th->getMessage() );
            return $this->error( 'An error occurred while updating the user account.', 500, $th->getMessage() );
        }
    }

    public function admin_update_user_aff_package( Request $request, $id ) {
        try {
            $validatedData = $request->validate( [
                'status' => [ 'required', 'in:ACTIVE,INACTIVE' ],
                'benefits' => 'required|array',
                'benefits.direct_referral_bonus.deposit_1' => 'required|numeric',
                'benefits.direct_referral_bonus.deposit_2' => 'required|numeric',
                'benefits.direct_referral_bonus.deposit_3' => 'required|numeric',
                'benefits.direct_referral_bonus.subsequent_deposits' => 'required|integer',
                'benefits.direct_referral_bonus.referred_person_bonus' => 'required|numeric',
                'benefits.direct_referral_bonus.deposit_bonus_up_to_times' => 'required|integer',
                'benefits.weekly_profit_from_direct.percentage' => 'required|numeric',
                'benefits.weekly_profit_from_direct.up_to_people' => 'required|integer',
                'benefits.indirect_referral_bonus.deposit_1' => 'required|numeric',
                'benefits.indirect_referral_bonus.levels' => 'required|integer',
                'benefits.monthly_profit_from_indirects.percentage' => 'required|numeric',
                'benefits.monthly_profit_from_indirects.up_to_people' => 'required|integer',
                'benefits.required_indirect_referrals' => 'required|integer',
                'benefits.premium_deposit_benefit' => 'required|integer',
                'benefits.min_referrals_required_for_cashout' => 'required|integer',
            ] );

            $user = User::findOrFail( $id );
            $benefits = $validatedData[ 'benefits' ];

            $package = $user->customAffiliatePackage;

            if ( !$package ) {
                $user->customAffiliatePackage()->create( [
                    'status' => $validatedData[ 'status' ],
                    'benefits' => $benefits
                ] );
            } else {
                $package->status = $validatedData[ 'status' ];
                $package->benefits = $benefits;
                $package->save();
            }

            return $this->success( $user->affiliatePackageOrDefault(), 'Custom affiliate package updated successfully.' );
        } catch ( \Illuminate\Validation\ValidationException $e ) {
            return $this->error( 'Validation failed: ' . $e->getMessage(), 422, $e->errors() );
        } catch ( \Exception $e ) {
            return $this->error( 'Something went wrong: ' . $e->getMessage(), 500, $e->getMessage() );
        }
    }

    public function add_transaction( Request $request ) {

        $validatedData = $request->validate( [
            'user_id' => 'required|exists:users,id',
            'type' => 'required|in:'. implode( ',', TransactionType::asArray() ),
            'scheme_id' => 'nullable|exists:investment_schemes,sid',
            'amount' => 'required|numeric|min:1',
            'remark' => 'nullable|string|max:255',
            'status' => 'required|in:'. implode( ',', StatusEnum::asArray() ),
            'interest_plan' => 'required_if:type,'.TransactionType::Investment.'|nullable|in:' . implode( ',', InterestPlan::asArray() ),
        ] );

        $user = User::findOrFail( $validatedData[ 'user_id' ] );
        $data[ 'amount_usd' ] = $validatedData[ 'amount' ];
        $data[ 'amount_crypto' ] = 0;

        $data[ 'payment_method' ][ 'description' ] = trim( $validatedData[ 'remark' ]??'' ) ?$validatedData[ 'remark' ]:'A ' . ucfirst( $validatedData[ 'type' ] ) . ' has been successfully credited to your account by an admin.';
        $data[ 'payment_method' ][ 'remark' ] =  $validatedData[ 'remark' ];
        $data[ 'payment_method' ][ 'is_added_by_admin' ] = true;

        if ( $validatedData[ 'type' ] == TransactionType::Investment ) {
            $scheme = InvestmentScheme::where( 'sid', $validatedData[ 'scheme_id' ] )->firstOrFail();
            $data[ 'scheme_id' ] = $scheme->id;
            $data[ 'interest_plan' ] = $validatedData[ 'interest_plan' ];
            $transaction = $user->createInvestmentDeposit( $data );
        } else if ( $validatedData[ 'type' ] == TransactionType::Bonus ) {
            $transaction = $user->addBonus( $data[ 'amount_usd' ], $data );
        } else if ( $validatedData[ 'type' ] == TransactionType::Commission ) {
            $transaction = $user->addCommission( $data[ 'amount_usd' ], $data );
        } else if ( $validatedData[ 'type' ] == TransactionType::Withdrawal ) {
            $transaction = $user->withdraw( $data[ 'amount_usd' ], $data[ 'amount_crypto' ], $data );
        }

        if ( $validatedData[ 'status' ] == StatusEnum::APPROVED && $transaction->status == StatusEnum::PENDING ) {
            $transaction = $user->approveTransaction( $transaction );
        }

        return $this->success( $transaction, 'Transaction created successfully.' );
    }

    public function delete_user_account( Request $request ) {
        $validatedData = $request->validate( [
            'user_id' => [ 'required', 'exists:users,id' ],
            'severity' => [ 'required', 'in:soft,permanent' ],
        ] );

        $user = User::withTrashed()->find( $validatedData[ 'user_id' ] );
        if ( !$user ) {
            return response()->json( [ 'error' => 'User not found' ], 404 );
        }

        if ( $validatedData[ 'severity' ] === 'soft' ) {
            $user->delete();
            // Perform a soft delete
        } else {
            $user->forceDelete();
            // Permanently delete the user
        }

        return $this->success( [], 'User account deleted successfully' );
    }

    /**
    * Add a new promo.
    */

    public function add_promo( Request $request ) {
        $validatedData = $request->validate( [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'discount_or_bonus' => 'required|numeric|min:0.01' ,
            'start_amount' => 'required|numeric|min:1',
            'type' => 'required|in:' . implode( ',', PromoType::asArray() ),
            'scheme_id' => 'required|exists:investment_schemes,id',
            'audience' => 'sometimes|in:'. implode( ',', AudienceGroupEnum::asArray() )
        ] );
        $scheme = InvestmentScheme::findOrFail( $validatedData[ 'scheme_id' ] );
        if ( $scheme->scheme_active_promo ) {
            return $this->error( 'This scheme already has an active promo. Modify or delete it to create new one...', 400 );
        }

        try {
            $promo = PromoService::createPromo( $validatedData );
            return $this->success( $promo, 'Promo added successfully.' );
        } catch ( \Throwable $th ) {
            Log::error( 'Error in AdminController -> add_promo: ' . $th->getMessage() );
            return $this->error( 'An error occurred while adding the promo.', 500, $th->getMessage() );
        }
    }

    /**
    * Update an existing promo.
    */

    public function update_promo( Request $request, $id ) {
        $validatedData = $request->validate( [
            'title' => 'sometimes|string|max:255',
            'description' => 'sometimes|string',
            'start_date' => 'sometimes|date',
            'end_date' => 'sometimes|date|after_or_equal:start_date',
            'discount_or_bonus' => 'sometimes|numeric|min:0.01' ,
            'start_amount' => 'sometimes|numeric|min:1',
            'type' => 'sometimes|in:' . implode( ',', PromoType::asArray() ),
            'scheme_id' => 'sometimes|nullable|exists:investment_schemes,id',
            'audience' => 'sometimes|in:'. implode( ',', AudienceGroupEnum::asArray() )
        ] );

        try {
            $promo = Promo::findOrFail( $id );
            $promo = PromoService::updatePromo( $promo->id, $validatedData );
            return $this->success( $promo, 'Promo updated successfully.' );
        } catch ( \Throwable $th ) {
            Log::error( 'Error in AdminController -> update_promo: ' . $th->getMessage() );
            return $this->error( 'An error occurred while updating the promo.', 500, $th->getMessage() );
        }
    }

    /**
    * Delete an existing promo.
    */

    public function delete_promo( $id ) {
        try {
            $promo = Promo::findOrFail( $id );
            $promo->delete();
            // Soft delete the promo, or use forceDelete() for permanent removal
            return $this->success( [], 'Promo deleted successfully.' );
        } catch ( \Throwable $th ) {
            Log::error( 'Error in AdminController -> delete_promo: ' . $th->getMessage() );
            return $this->error( 'An error occurred while deleting the promo.', 500, $th->getMessage() );
        }
    }

    public function exportUsers( Request $request ) {
        $request->validate( [
            'email' => 'required|email',
            'paying_customers' => [ 'required', 'boolean' ],
            'non_paying_customers' => [ 'required', 'boolean' ],
            'new_users' => [ 'required', 'boolean' ],
        ] );

        $filteredUsers = AdminService::getFilteredUsers( $request->all( [ 'paying_customers', 'non_paying_customers', 'new_users' ] ), true );

        if ( empty( $filteredUsers[ 'attachments' ] ) ) {
            return $this->error( 'No users found to export.', 404 );
        }

        Mail::to( $request->email )->send( new CsvExportMail( $filteredUsers[ 'attachments' ] ) );

        // Cleanup files
        foreach ( $filteredUsers[ 'attachments' ] as $attachment ) {
            unlink( $attachment[ 'path' ] );
        }

        return $this->success( [], 'CSV export(s) have been sent to ' . $request->email );
    }

    public function broadcast_email( Request $request ) {

        $request->validate( [
            'email_content' => 'required|string',
            'email_subject' => 'required|string',
            'paying_customers' => [ 'required', 'boolean' ],
            'non_paying_customers' => [ 'required', 'boolean' ],
            'new_users' => [ 'required', 'boolean' ],
        ] );

        $delayInSeconds = 2;
        $index = 0;
        $usersCount = 1;

        if (
            system_setting( SystemSettings::ENABLED_2FA, false )  && ( ! $request->user()->two_factor_verified_at ||
            $request->user()->two_factor_verified_at->diffInSeconds( now() ) > 5 )
        ) {
            return $this->error( '2FA expired. Please re-verify.', 403 );
        }

        try {
            if ( $request->input( 'email' ) ) {
                $user = User::where( 'email', $request->input( 'email' ) )->firstOrFail();
                BroadcastJob::dispatch(
                    to: $user->email,
                    subject: $request->input( 'email_subject' ),
                    content: $request->input( 'email_content' ),
                    userData: $user->toArray()
                )->delay( now()->addSeconds( $index * $delayInSeconds ) );

            } else {
                $filteredUsers = AdminService::getFilteredUsers( $request->all( [ 'paying_customers', 'non_paying_customers', 'new_users' ] ) );
                if ( count( $filteredUsers[ 'users' ] ) <= 0 ) {
                    return $this->error( 'No users found to send email to.', 404 );
                }

                foreach ( $filteredUsers[ 'users' ] as $user ) {
                    BroadcastJob::dispatch(
                        to: $user[ 'email' ],
                        subject: $request->input( 'email_subject' ),
                        content: $request->input( 'email_content' ),
                        userData: $user
                    )->delay( now()->addSeconds( $index * $delayInSeconds ) );
                    $index++;
                }

                $usersCount += count( $filteredUsers[ 'users' ] );
            }
        } catch ( \Throwable $th ) {
            Log::error( 'Something went wrong -> ' . $th->getMessage() );
            return $this->error( 'Something went wrong -> ' . $th->getMessage(), 500 );
        }

        return $this->success( [], 'Your email broadcast has been submitted... and witll be sent to '.$usersCount . ' Users' );
    }

    public function admin_create_charge_fee_transactions( Request $request ) {
        $validated = $request->validate( [
            'emails' => 'required|array',
            'emails.*' => 'required|email',
            'amount' => 'required|numeric|min:0.01',
            'description' => 'required|string|max:255',
            'send_email' => 'required|boolean',
            'fee_variant' => 'required|in:'.implode( ',', FeeVariantEnum::getValues() ),
        ] );

        $admin = $request->user();

        try {
            $results = [];
            $failed = [];

            foreach ( $validated[ 'emails' ] as $email ) {
                try {
                    // Find user by email
                    $user = User::where( 'email', $email )->first();

                    if ( !$user ) {
                        $failed[ $email ] = 'User not found';
                        continue;
                    }
                    $methodInfo[ 'description' ] = $validated[ 'description' ];
                    $methodInfo[ 'send_email' ] = $validated[ 'send_email' ];
                    // $methodInfo[ 'created_by' ] = '';
                    $methodInfo[ 'fee_variant' ] = $validated[ 'fee_variant' ];

                    $trasanction = $user->createDepositChargeTransaction(
                        $validated[ 'amount' ],
                        $methodInfo,

                    );
                    $results[ $email ] = 'Success';

                } catch ( \Exception $e ) {
                    $failed[ $email ] = 'Error: ' . $e->getMessage();
                }
            }

            $response = [
                'success' => true,
                'message' => 'Fee charges processed successfully',
                'results' => $results,
                'failed' => $failed,
                'total_charged' => count( $results ) * $validated[ 'amount' ],
                'success_count' => count( $results ),
                'failed_count' => count( $failed )
            ];

            return $this->success( $response, $response[ 'message' ] );

        } catch ( \Exception $e ) {
            // Log error
            Log::error( 'Fee charge error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ] );

            return $this->error( 'Error processing fee charges: ' . $e->getMessage(), 500 );
        }
    }

    // 2. Actually CREATE a new tier ( this is the one you were missing! )

    public function admin_create_maturity_bonus_tier( Request $request ) {
        $validated = $request->validate( [
            'investment_scheme_id' => 'nullable|exists:investment_schemes,id',
            'title'              => 'required|string|max:255|unique:investment_maturity_bonus_tiers,title',
            'description'        => 'nullable|string',
            'min_amount'         => 'required|numeric|min:0',
            'max_amount'         => 'nullable|numeric|gte:min_amount',
            'bonus_percentage'   => 'required|numeric|min:0|max:999.9999',
            'interest_percentage_on_investment'   => 'required|numeric|min:0|max:999.9999',
            'starts_at'          => 'nullable|date',
            'ends_at'            => 'nullable|date|after_or_equal:starts_at',
            'is_active'          => 'sometimes|boolean',
        ] );

        $tier = InvestmentMaturityBonusTier::create( [
            'investment_scheme_id' => $request->filled( 'investment_scheme_id' ) ? $request->investment_scheme_id : null,
            'title'              => $request->title,
            'description'        => $request->description,
            'min_amount'         => $request->min_amount,
            'max_amount'         => $request->max_amount,
            'bonus_percentage'   => $request->bonus_percentage,
            'interest_percentage_on_investment'   => $request->interest_percentage_on_investment,
            'starts_at'          => $request->starts_at,
            'ends_at'            => $request->ends_at,
            'is_active'          => $request->boolean( 'is_active', true ),
        ] );

        return $this->success( [ 'tier'    => $tier->load( 'plan' ) ], 'Bonus tier created successfully!', 201 );
    }

    public function admin_update_maturity_bonus_tier( Request $request, InvestmentMaturityBonusTier $bonusTier ) {

        $request->validate( [
            'investment_scheme_id' => 'nullable|exists:investment_schemes,id',
            'title'              => 'required|string|max:255',
            'description'        => 'nullable|string',
            'min_amount'         => 'required|numeric|min:0',
            'max_amount'        => 'nullable|numeric|gte:min_amount',
            'bonus_percentage'   => 'required|numeric|min:0|max:999.9999',
            'interest_percentage_on_investment'   => 'required|numeric|min:0|max:999.9999',
            'starts_at'          => 'nullable|date',
            'ends_at'            => 'nullable|date|after_or_equal:starts_at',
            'is_active'          => 'sometimes|boolean',
        ] );

        $bonusTier->update( [
            'investment_scheme_id' => $request->filled( 'investment_scheme_id' ) ? $request->investment_scheme_id : null,
            'title'              => $request->title,
            'description'        => $request->description,
            'min_amount'         => $request->min_amount,
            'max_amount'         => $request->max_amount,
            'bonus_percentage'   => $request->bonus_percentage,
            'interest_percentage_on_investment'   => $request->interest_percentage_on_investment,
            'starts_at'          => $request->starts_at,
            'ends_at'            => $request->ends_at,
            'is_active'          => $request->boolean( 'is_active', $bonusTier->is_active ),
        ] );

        return $this->success( [ 'tier'    => $bonusTier->load( 'plan' ) ], 'Maturity bonus tier updated successfully!' );
    }
}
