<?php

namespace App\Console\Commands;

use App\Enums\StatusEnum;
use App\Models\Rental;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ProcessRentalsRevenue extends Command
{
    protected $signature = 'app:process-rentals-revenue';
    protected $description = 'Process matured rentals revenue';

    public function handle() {
        $chunkSize = 100;
        $this->info("Starting to process matured rentals in chunks of $chunkSize...");
        Rental::where('next_revenue_at', '<=', now())
             ->where('status', StatusEnum::APPROVED)
            ->chunk($chunkSize, function ($rentals) {
                foreach ($rentals as $rental) {
                    $this->processRentalRevenue($rental);
                }
                $this->info("Processed a chunk of rentals...");
            });

        $this->info("Completed processing of matured rentals.");
    }

    private function processRentalRevenue(Rental $rental) {
        DB::beginTransaction();

        try {
            // Calculate the revenue
            $revenueGenerated = $rental->calculateRevenue();

            // Schedule the next revenue processing time (1 minute later)
            $nextRevenueAt = now()->addMinute(); // Instead of 0.0166667

            // Update rental details
            $rental->update([
                'last_revenue_processed_at' => now(),
                'next_revenue_at' => $nextRevenueAt
            ]);

            // Increment total revenue
            $rental->increment('revenue_generated', $revenueGenerated);

            // Update robot's last revenue timestamp
            $rental->robot->update(['last_revenue_at' => now()]);

            DB::commit();

            $this->info("Processed revenue for Rental ID {$rental->id} with generated revenue of {$revenueGenerated}.");
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error("Error processing rental revenue for Rental ID {$rental->id}: {$e->getMessage()}", [
                'rental_id' => $rental->id,
                'exception' => $e
            ]);

            $this->error("Error processing rental revenue for Rental ID {$rental->id}. Check logs for details.");
        }
    }

}
